#!/usr/bin/env python3

import hashlib
import os
import sys


if sys.version_info[0] < 3:
    # Python 2.
    from urllib import urlretrieve
    from urlparse import urlsplit
else:
    # Python 3.
    from urllib.request import urlretrieve
    from urllib.parse import urlsplit


DOWNLOADS_DIR = os.path.join('.cache', 'downloads')


def download(url, sha1=None, filename=None, downloads_dir=DOWNLOADS_DIR):
    if filename is None:
        filename = os.path.basename(urlsplit(url).path)
    dst = os.path.join(downloads_dir, filename)
    dst_dir = os.path.dirname(dst)
    if not os.path.exists(dst_dir):
        os.makedirs(dst_dir)
    retries = 0
    while retries < 2:
        if sha1 is None or not os.path.exists(dst):
            retries += 1
            try:
                urlretrieve(url, dst)
            except Exception as e:
                print('error', e)
                continue
        if sha1 is None:
            break
        h = hashlib.sha1()
        with open(dst, 'rb') as fp:
            while True:
                d = fp.read(4 * 1024 * 1024)
                if not d:
                    break
                h.update(d)
        if h.hexdigest() == sha1:
            break
        print('sha1 does not match: %s instead of %s' % (h.hexdigest(), sha1))
        os.unlink(dst)
    assert os.path.exists(dst), 'could not successfully retrieve %s' % url
    return dst


if __name__ == '__main__':
    args = sys.argv[1:]
    url = args.pop(0)
    sha1 = None
    filename = None
    if args:
        sha1 = args.pop(0) or None
    if args:
        filename = args.pop(0)
    print(download(url, sha1, filename))
