/***********************************************************************************
* Smooth Tasks
* Copyright (C) 2009 Michal Odstrčil <michalodstrcil@gmail.com>
* Copyright (C) 2009 Mathias Panzenböck <grosser.meister.morti@gmx.net> (minor contributions)
* Copyright (C) 2009-2010 Toni Dietze <smooth-tasks@derflupp.e4ward.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#include "SmoothTasks/LimitSqueezeTaskbarLayout.h"

#include "Global.h"

#include <cmath>

namespace SmoothTasks {

void LimitSqueezeTaskbarLayout::setSqueezeRatio(qreal squeezeRatio) {
	if (m_squeezeRatio != squeezeRatio) {
		m_squeezeRatio = squeezeRatio;
		invalidate();
	}
}

void   LimitSqueezeTaskbarLayout::setPreferGrouping(bool preferGrouping) {
	if (m_preferGrouping != preferGrouping) {
		m_preferGrouping = preferGrouping;
		invalidate();
	}
}

int LimitSqueezeTaskbarLayout::optimumCapacity() const {
	const QRectF effectiveRect(effectiveGeometry());
	
	const bool isVertical = orientation() == Qt::Vertical;
	const qreal availableWidth = isVertical ? effectiveRect.height() : effectiveRect.width();
	
	const qreal spacing = this->spacing();
	
	const int rows = m_preferGrouping ? minimumRows() : maximumRows();
	
	return static_cast<qreal>(rows) * (availableWidth + spacing) / (m_squeezeRatio * averagePreferredItemRdWidthStatic(m_cellHeight, true) + spacing);
}

void LimitSqueezeTaskbarLayout::updateLayoutStatic() {
	const bool isVertical = orientation() == Qt::Vertical;
	
	const QList<TaskbarItem*>& items = this->items();
	const int N = items.size();
	
	const QRectF effectiveRect(effectiveGeometry());
	const qreal availableWidth  = isVertical ? effectiveRect.height() : effectiveRect.width();
	const qreal availableHeight = isVertical ? effectiveRect.width()  : effectiveRect.height();
	const qreal spacing         = this->spacing();
	
#define CELL_HEIGHT(ROWS) (((availableHeight + spacing) / ((qreal) (ROWS))) - spacing)
	
	// if there is nothing to layout fill in some dummy data and leave
	if (N == 0) {
		m_rows = minimumRows();
		m_cellHeight = CELL_HEIGHT(m_rows);
		m_rowInfos.clear();
		m_compression = 1.0;
		return;
	}
	
	QList<RowInfo> rowInfos;
	
	int rows = std::ceil(m_squeezeRatio * (comulativePreferredItemRdWidthStatic(CELL_HEIGHT(minimumRows())) + (qreal) (N - minimumRows()) * spacing) / availableWidth);
	if (rows < minimumRows()) rows = minimumRows();
	if (rows > maximumRows()) rows = maximumRows();
	
	QVector<qreal> itemRdWidths(N);
	{
		const qreal cellHeight = CELL_HEIGHT(rows);
		int i = 0;
		foreach (TaskbarItem* item, items) {
			itemRdWidths[i] = item->item->preferredRdSizeStatic(cellHeight).width();
			++i;
		}
	}
	qreal itemRdWidthsSum = ::SmoothTasks::squeezeWidths(itemRdWidths, (qreal) rows * availableWidth - (qreal) (N - rows) * spacing);
	
	{
		const int maxRowIndex = maximumRows() - 1;
		qreal currentRdWidth = -spacing;
		int currentRow = 0;
		int currentStartIndex = 0;
		for (int i = 0; i < N; ++i) {
			currentRdWidth += itemRdWidths[i] + spacing;
			if (currentRdWidth >= availableWidth) {
				if (currentRdWidth - availableWidth > 0.5 * itemRdWidths[i]  &&  currentStartIndex != i) {
					// much overhang -> put on next row
					rowInfos.append(RowInfo(currentStartIndex, i));
					currentStartIndex = i;
					currentRdWidth = itemRdWidths[i];
				} else {
					// little overhang -> put on this row
					rowInfos.append(RowInfo(currentStartIndex, i + 1));
					currentStartIndex = i + 1;
					currentRdWidth = -spacing;
				}
				++currentRow;
				if (currentRow == maxRowIndex) break;
			}
		}
		if (currentStartIndex < N) {
			rowInfos.append(RowInfo(currentStartIndex, N));
		}
	}
	
	// if we assumed expanded there still might be empty row
	// therefore just scale up the layout (maybe do only this and not the other way of row removal)
	m_rows = qMax(minimumRows(), rowInfos.size());
	m_cellHeight = CELL_HEIGHT(m_rows);
	m_rowInfos = rowInfos;
	updateItemsRowCache();
	m_compression = (itemRdWidthsSum + (qreal) (N - m_rows) * spacing) / (comulativePreferredItemRdWidthStatic(m_cellHeight) + (qreal) (N - m_rows) * spacing);
#undef CELL_HEIGHT
}

} // namespace SmoothTasks
