///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __MULTI_FILE_WRITER_H
#define __MULTI_FILE_WRITER_H

#include <core/Core.h>
#include "AbstractFileColumnWriter.h"

namespace AtomViz {

/**
 * \brief Abstract base class for atom file format writers that can
 *        writer multiple atoms files as a movie.  
 * 
 * This abstract base class is meant as basis for specialized file format
 * writers that should be able to write multiple files to a directory
 * each containing a single timestep.
 * 
 * Alternatively this base class supports multi-timestep files. These kind
 * of files contain more than one timestep in a single file.
 * 
 * \author Alexander Stukowski 
 */ 
class ATOMVIZ_DLLEXPORT MultiFileWriter : public AbstractFileColumnWriter
{
protected:

	/// \brief Constructs a new instance of this class. 
	/// \param isLoading Specifies whether the object's data fields will be initialized from the 
	///                  data stored in a scene file after the instance has been created.
	MultiFileWriter(bool isLoading = false) : AbstractFileColumnWriter(isLoading),
		_useWildcardFilename(false), _startFrame(0), _endFrame(-1) {
		INIT_PROPERTY_FIELD(MultiFileWriter, _useWildcardFilename);
		INIT_PROPERTY_FIELD(MultiFileWriter, _wildcardFilename);
		INIT_PROPERTY_FIELD(MultiFileWriter, _startFrame);		
		INIT_PROPERTY_FIELD(MultiFileWriter, _endFrame);
	}

public:

	/// \brief Sets whether the writer should produce multiple data files in the directory 
	///        using a wild-card pattern.
	/// \sa setWildcardFilename(), useWildcardFilename() 
	void setUseWildcardFilename(bool enable) { this->_useWildcardFilename = enable; }
	
	/// \brief Returns whether the writer produces multiple data files in the directory
	///        using a wild-card pattern.
	/// \sa setWildcardFilename(), setUseWildcardFilename()
	bool useWildcardFilename() const { return this->_useWildcardFilename; }
	
	/// \brief Sets the wild-card pattern that is used to generate the filenames when writing
	///        multiple atom files.
	///
	/// The wild-card filename must contain the special character * that will be replaced by the
	/// frame number. 
	///
	/// The wild-card filename is only used if the feature has been enabled using setUseWildcardFilename().
	///
	/// \sa setWildcardFilename(), useWildcardFilename(), wildcardFilename() 
	void setWildcardFilename(const QString& filename) { this->_wildcardFilename = filename; }
	
	/// \brief Returns the wild-card pattern used to generate multiple files.
	/// \sa setWildcardFilename()
	const QString& wildcardFilename() const { return this->_wildcardFilename; }

	/// \brief Sets the start of the animation interval that should be exported to the output file.
	/// \sa startFrame(), setEndFrame()
	void setStartFrame(int frame) { _startFrame = frame; }

	/// \brief Returns the first frame of the animation interval that will be exported to the output file.
	/// \sa setStartFrame(), endFrame()
	TimeTicks startFrame() const { return _startFrame; }

	/// \brief Sets the end of the animation interval that should be exported to the output file.
	/// \sa endFrame(), setStartFrame()
	void setEndFrame(int frame) { _endFrame = frame; }

	/// \brief Returns the last frame of the animation interval that will be exported to the output file.
	/// \sa setEndFrame(), startFrame()
	TimeTicks endFrame() const { return _endFrame; }
	
	// From AtomsFileWriter:

	/// \brief Sets the name of the output file for this writer.
	virtual void setOutputFile(const QString& filename) {
		// Use filename as default wild-card pattern.
		if(wildcardFilename().isEmpty()) {
			if(!filename.contains('*'))
				setWildcardFilename(QFileInfo(filename).fileName() + ".*");
			else
				setWildcardFilename(QFileInfo(filename).fileName());
		}
		AbstractFileColumnWriter::setOutputFile(filename);
	}
	
	/// \brief Prepares the writer based on the scene to be exported.
	virtual bool prepareScene(DataSet* dataset, bool suppressDialogs) {
		
		// Use the current time as default export interval if no interval has been set before.
		if(startFrame() > endFrame()) {
			int currentFrame = timeToFrame(dataset->animationSettings()->time(), dataset->animationSettings()->ticksPerFrame());	
			setStartFrame(currentFrame);
			setEndFrame(currentFrame);
		}
		
		return AbstractFileColumnWriter::prepareScene(dataset, suppressDialogs); 
	}	

	/// \brief Exports the atoms in the given scene to the output file.
	virtual bool exportAtoms(DataSet* dataset, bool suppressDialogs = false);

public:

	Q_PROPERTY(bool useWildcardFilename READ useWildcardFilename WRITE setUseWildcardFilename)
	Q_PROPERTY(QString wildcardFilename READ wildcardFilename WRITE setWildcardFilename)

	Q_PROPERTY(int startFrame READ startFrame WRITE setStartFrame)
	Q_PROPERTY(int endFrame READ endFrame WRITE setEndFrame)

protected:

	/// \brief Writes the output file.
	/// \param filepath The path of the output file.
	/// \param dataset The scene containing the AtomsObject to be exported.
	/// \param exportFrames A list of animation frames that should be written to the output file. This list contains at least one frame.
	/// \param suppressDialogs Specifies whether any dialogs or message boxes shown by the writer should be supressed during preparation.
	///                        This parameter is set to true when the parser is invoked from a script to not interrupt the export process. 
	/// \throws Exception on error.	
	/// \return \a false when the operation has been canceled by the user; \a true on success.
	///
	/// This method is called by exportAtoms() to do the actual writing of the output file.
	virtual bool writeAtomsFile(const QString& filepath, DataSet* dataset, const QVector<TimeTicks>& exportFrames, bool suppressDialogs) = 0;
	
private:

	/// Indicates that the writer should produce a new file in the same directory for each timestep.
	PropertyField<bool> _useWildcardFilename;
	
	/// The wild-card name that is used to generate the output filenames.
	PropertyField<QString> _wildcardFilename;

	/// The first movie frame that should be exported.
	PropertyField<int> _startFrame;

	/// The last movie frame that should be exported.
	PropertyField<int> _endFrame;
	
private:
	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(MultiFileWriter)
	DECLARE_PROPERTY_FIELD(_useWildcardFilename)
	DECLARE_PROPERTY_FIELD(_wildcardFilename)
	DECLARE_PROPERTY_FIELD(_startFrame)
	DECLARE_PROPERTY_FIELD(_endFrame)
};

};	// End of namespace AtomViz

#endif // __MULTI_FILE_WRITER_H
