/*
 *  Copyright (c) 2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "../Transform.h"

#include <cstdlib>
#include "../Region.h"

double rand_f()
{
  return rand() / double(RAND_MAX) * 1000;
}

class TestNoneTransform : public GTLTest::Case {
  public:
    TestNoneTransform() : GTLTest::Case("NoneTransform")
    {
    }
    virtual void runTest()
    {
      double x = rand_f();
      double y = rand_f();
      double xo, yo;
      GTLCore::Transform transfo;
      transfo.map(x,y, xo, yo);
      GTLTEST_CHECK_NEAR_EQUAL( x, xo);
      GTLTEST_CHECK_NEAR_EQUAL( y, yo);
      
      GTLCore::Transform invTransfo1 = transfo.invert();
      invTransfo1.map(xo,yo, xo, yo);
      GTLTEST_CHECK_NEAR_EQUAL( x, xo);
      GTLTEST_CHECK_NEAR_EQUAL( y, yo);
    }
};

class TestRotationTransform  : public GTLTest::Case {
  public:
    TestRotationTransform() : GTLTest::Case("RotationTransform")
    {
    }
    virtual void runTest()
    {
      double x = 1.0;
      double y = 0.0;
      double xo, yo;
      
      // Rotation by PI
      GTLCore::Transform transfo1;
      transfo1.rotate(M_PI);
      transfo1.map(x,y, xo, yo);
      GTLTEST_CHECK_NEAR_EQUAL( xo, -1.0);
      GTLTEST_CHECK_NEAR_EQUAL( yo, 0.0);

      GTLCore::Transform invTransfo1 = transfo1.invert();
      invTransfo1.map(xo,yo, xo, yo);
      GTLTEST_CHECK_NEAR_EQUAL( x, xo);
      GTLTEST_CHECK_NEAR_EQUAL( y, yo);
      
      // Rotation by PI/2
      GTLCore::Transform transfo2;
      transfo2.rotate(0.5 * M_PI);
      transfo2.map(x,y, xo, yo);
      GTLTEST_CHECK_NEAR_EQUAL( xo, 0.0);
      GTLTEST_CHECK_NEAR_EQUAL( yo, 1.0);
      
      GTLCore::Transform invTransfo2 = transfo2.invert();
      invTransfo2.map(xo,yo, xo, yo);
      GTLTEST_CHECK_NEAR_EQUAL( x, xo);
      GTLTEST_CHECK_NEAR_EQUAL( y, yo);

      // Rotation by 3 PI/2
      GTLCore::Transform transfo3;
      transfo3.rotate(1.5 * M_PI);
      transfo3.map(x,y, xo, yo);
      GTLTEST_CHECK_NEAR_EQUAL( xo, 0.0);
      GTLTEST_CHECK_NEAR_EQUAL( yo, -1.0);
      
      GTLCore::Transform invTransfo3 = transfo3.invert();
      invTransfo3.map(xo,yo, xo, yo);
      GTLTEST_CHECK_NEAR_EQUAL( x, xo);
      GTLTEST_CHECK_NEAR_EQUAL( y, yo);
    }
};

class TestMultipleRotationTransform  : public GTLTest::Case {
  public:
    TestMultipleRotationTransform() : GTLTest::Case("MultipleRotationTransform")
    {
    }
    virtual void runTest()
    {
      double x = 1.0;
      double y = 0.0;
      double xo, yo;
      
      // Rotation by 2 * PI / 2
      GTLCore::Transform transfo1;
      transfo1.rotate(0.5 * M_PI);
      transfo1.rotate(0.5 * M_PI);
      transfo1.map(x,y, xo, yo);
      GTLTEST_CHECK_NEAR_EQUAL( xo, -1.0);
      GTLTEST_CHECK_NEAR_EQUAL( yo, 0.0);
      
      GTLCore::Transform invTransfo1 = transfo1.invert();
      invTransfo1.map(xo,yo, xo, yo);
      GTLTEST_CHECK_NEAR_EQUAL( x, xo);
      GTLTEST_CHECK_NEAR_EQUAL( y, yo);
    }
};

class TestTranslationTransform  : public GTLTest::Case {
  public:
    TestTranslationTransform() : GTLTest::Case("TranslationTransform")
    {
    }
    virtual void runTest()
    {
      double x = rand_f();
      double y = rand_f();
      double xo, yo;
      
      // Rotation by 2 * PI / 2
      GTLCore::Transform transfo1;
      transfo1.translate(-3, 500);
      transfo1.map(x,y, xo, yo);
      GTLTEST_CHECK_NEAR_EQUAL( x - 3, xo);
      GTLTEST_CHECK_NEAR_EQUAL( y + 500, yo);
      
      GTLCore::Transform invTransfo1 = transfo1.invert();
      invTransfo1.map(xo,yo, xo, yo);
      GTLTEST_CHECK_NEAR_EQUAL( x, xo);
      GTLTEST_CHECK_NEAR_EQUAL( y, yo);
    }
};

class TestScaleTransform  : public GTLTest::Case {
  public:
    TestScaleTransform() : GTLTest::Case("ScaleTransform")
    {
    }
    virtual void runTest()
    {
      double x = rand_f();
      double y = rand_f();
      double xo, yo;
      
      // Rotation by 2 * PI / 2
      GTLCore::Transform transfo1;
      transfo1.scale(-0.5, 2.0);
      transfo1.map(x,y, xo, yo);
      GTLTEST_CHECK_NEAR_EQUAL( -0.5 * x, xo);
      GTLTEST_CHECK_NEAR_EQUAL( 2 * y, yo);
      
      GTLCore::Transform invTransfo1 = transfo1.invert();
      invTransfo1.map(xo,yo, xo, yo);
      GTLTEST_CHECK_NEAR_EQUAL( x, xo);
      GTLTEST_CHECK_NEAR_EQUAL( y, yo);
    }
};

class TestRegionMapTransform  : public GTLTest::Case {
  public:
    TestRegionMapTransform() : GTLTest::Case("RegionMapTransform")
    {
    }
    virtual void runTest()
    {
      // Translation
      GTLCore::RegionF region(3,4,10,20);
      GTLCore::Transform transfo1;
      transfo1.translate(-3, 500);
      GTLCore::RegionF regiont = transfo1.map(region);
      GTLTEST_CHECK_NEAR_EQUAL( region.x() - 3, regiont.x());
      GTLTEST_CHECK_NEAR_EQUAL( region.y() + 500, regiont.y());
      GTLTEST_CHECK_NEAR_EQUAL( region.columns(), regiont.columns() );
      GTLTEST_CHECK_NEAR_EQUAL( region.rows(), regiont.rows() );
      
      // Rotation pi/2
      region = GTLCore::RegionF(-5,-5,10,10);
      GTLCore::Transform transfo2;
      transfo2.rotate(M_PI_2);
      regiont = transfo2.map(region);
      GTLTEST_CHECK_NEAR_EQUAL( region.x(), regiont.x());
      GTLTEST_CHECK_NEAR_EQUAL( region.y(), regiont.y());
      GTLTEST_CHECK_NEAR_EQUAL( region.columns(), regiont.columns() );
      GTLTEST_CHECK_NEAR_EQUAL( region.rows(), regiont.rows() );
      
    }
};

class TestScaleCenterTransform : public GTLTest::Case {
  public:
    TestScaleCenterTransform() : GTLTest::Case("TestScaleCenterTransform")
    {
    }
    virtual void runTest()
    {
      double rCenterX = 100.0;
      double rCenterY = 120.0;
      GTLCore::Transform transfo;
      transfo.scale( 0.5, 0.5 );
      GTLCore::Transform transfo1;
      transfo1.translate(-rCenterX, -rCenterY);
      GTLCore::Transform transfo2;
      transfo2.translate(rCenterX, rCenterY);
      
      // Test without composition
      double x,y,x1,y1;
      transfo1.map(rCenterX, rCenterY, x,y);
      transfo.map(x,y,x1,y1);
      transfo2.map(x1,y1,x,y);
      GTLTEST_CHECK_NEAR_EQUAL( rCenterX, x);
      GTLTEST_CHECK_NEAR_EQUAL( rCenterY, y);
      transfo1.map(110.0, 130.0, x,y);
      transfo.map(x,y,x1,y1);
      transfo2.map(x1,y1,x,y);
      GTLTEST_CHECK_NEAR_EQUAL( 105.0, x);
      GTLTEST_CHECK_NEAR_EQUAL( 125.0, y);
      
      // Test with composition
      transfo = transfo * transfo1;
      transfo = transfo2 * transfo;
      transfo.map(rCenterX, rCenterY, x, y);
      GTLTEST_CHECK_NEAR_EQUAL( rCenterX, x);
      GTLTEST_CHECK_NEAR_EQUAL( rCenterY, y);
      transfo.map(110.0, 130.0, x, y);
      GTLTEST_CHECK_NEAR_EQUAL( 105.0, x);
      GTLTEST_CHECK_NEAR_EQUAL( 125.0, y);
      
      // Test inversion
      transfo = transfo.invert();
      transfo.map(rCenterX, rCenterY, x, y);
      GTLTEST_CHECK_NEAR_EQUAL( rCenterX, x);
      GTLTEST_CHECK_NEAR_EQUAL( rCenterY, y);
      transfo.map(105.0, 125.0, x, y);
      GTLTEST_CHECK_NEAR_EQUAL( 110.0, x);
      GTLTEST_CHECK_NEAR_EQUAL( 130.0, y);
      
    }
};

class TestRotationCenterTransform : public GTLTest::Case {
  public:
    TestRotationCenterTransform() : GTLTest::Case("RotationCenterTransform")
    {
    }
    virtual void runTest()
    {
      double rCenterX = 100.0;
      double rCenterY = 120.0;
      GTLCore::Transform transfo;
      transfo.rotate( M_PI_2 );
      GTLCore::Transform transfo1;
      transfo1.translate(-rCenterX, -rCenterY);
      GTLCore::Transform transfo2;
      transfo2.translate(rCenterX, rCenterY);
      
      // Test without composition
      double x,y,x1,y1;
      transfo1.map(rCenterX, rCenterY, x,y);
      transfo.map(x,y,x1,y1);
      transfo2.map(x1,y1,x,y);
      GTLTEST_CHECK_NEAR_EQUAL( rCenterX, x);
      GTLTEST_CHECK_NEAR_EQUAL( rCenterY, y);
      transfo1.map(105.0, 120.0, x,y);
      transfo.map(x,y,x1,y1);
      transfo2.map(x1,y1,x,y);
      GTLTEST_CHECK_NEAR_EQUAL( 100.0, x);
      GTLTEST_CHECK_NEAR_EQUAL( 125.0, y);
      
      // Test with composition
      transfo = transfo * transfo1;
      transfo = transfo2 * transfo;
      transfo.map(rCenterX, rCenterY, x, y);
      GTLTEST_CHECK_NEAR_EQUAL( rCenterX, x);
      GTLTEST_CHECK_NEAR_EQUAL( rCenterY, y);
      transfo.map(105.0, 120.0, x, y);
      GTLTEST_CHECK_NEAR_EQUAL( 100.0, x);
      GTLTEST_CHECK_NEAR_EQUAL( 125.0, y);
      
      // Test inversion
      transfo = transfo.invert();
      transfo.map(rCenterX, rCenterY, x, y);
      GTLTEST_CHECK_NEAR_EQUAL( rCenterX, x);
      GTLTEST_CHECK_NEAR_EQUAL( rCenterY, y);
      transfo.map(100.0, 125.0, x, y);
      GTLTEST_CHECK_NEAR_EQUAL( 105.0, x);
      GTLTEST_CHECK_NEAR_EQUAL( 120.0, y);
      
    }
};

class TestTransform : public GTLTest::Suite {
  public:
    TestTransform() : GTLTest::Suite("Transform")
    {
      addCase(new TestNoneTransform);
      addCase(new TestRotationTransform);
      addCase(new TestMultipleRotationTransform);
      addCase(new TestTranslationTransform);
      addCase(new TestScaleTransform);
      addCase(new TestRegionMapTransform);
      addCase(new TestScaleCenterTransform);
      addCase(new TestRotationCenterTransform);
    }
};

