/*
 * Copyright (c) 2007,2008, 2009 INRIA, UDcast
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Mohamed Amine Ismail <amine.ismail@sophia.inria.fr>
 *                              <amine.ismail@udcast.com>
 */

#ifndef WIMAX_DEFAULT_TRACES_H
#define WIMAX_DEFAULT_TRACES_H

namespace ns3
{

/**
 * \brief These represent default traces, providing a number of
 * parameters for each SNR value.
 *
 * The parameters provided are (in order):
 *
 * -#  The SNR value,
 * -#  The bit error rate BER,
 * -#  The block error rate BlcERm,
 * -#  The standard deviation on block error rate,
 * -#  The lower bound confidence interval for a given modulation, and
 * -#  The upper bound confidence interval for a given modulation.
 *
 * It is made of 7 tables: one for each modulation and coding scheme.
 * Each table is made of 6 columns, as listed above.
 *
 * These traces are generated by an external OFDM simulator: It is based
 * on an external mathematics and signal processing library IT++ and includes:
 *
 * *  A random block generator,
 * *  A Reed Solomon (RS) coder,
 * *  A convolutional coder,
 * *  an interleaver,
 * *  A 256 FFT-based OFDM modulator,
 * *  A multi-path channel simulator, and
 * *  An equalizer.
 *
 * The multipath channel is simulated using the TDL_channel class of
 * the IT++ library.
 *
 *
 * The file format should be as follows
 *
 * SNR_value1   BER  Blc_ER  STANDARD_DEVIATION  CONFIDENCE_INTERVAL1  CONFIDENCE_INTERVAL2
 * SNR_value2   BER  Blc_ER  STANDARD_DEVIATION  CONFIDENCE_INTERVAL1  CONFIDENCE_INTERVAL2
 *  ...          ...  ...     ...                 ...                   ...
 *
 *
 *  Additional notes: from Esteban Municio   <esteban.municio@urjc.es>
 *
 *  This trace file has been edited for the 16 QAM 3/4 modulation (modulation4) since the original
 * SNR traces for this modulation were not long enough, giving inflated BER and BLER figures for
 * typical SNR values. Due to the impossibility to get the original IT++ environment used for the
 * original traces by the previous authors, the traces of modulation4 have been generated using a
 * simple IT++ program simulating the theoretical model of a AWGN channel given
 * BER=3/2*erfc(sqrt(k*0.1*(Eb_N0_dB)));
 *
 * \relates ns3::SNRToBlockErrorRateManager
 * \hideinitializer
 * @{
 */

static const double modulation0[6][29] = {

    {0.00, 0.10, 0.20, 0.30, 0.40, 0.50, 0.60, 0.70, 0.80, 0.90, 1.00, 1.10, 1.20, 1.30, 1.40,
     1.50, 1.60, 1.70, 1.80, 1.90, 2.00, 2.10, 2.20, 2.30, 2.40, 2.50, 2.60, 2.70, 2.80},

    {0.50798, 0.50862, 0.50517, 0.49325, 0.45741, 0.39430, 0.30345, 0.20492, 0.12126, 0.06741,
     0.03429, 0.01710, 0.00865, 0.00419, 0.00223, 0.00098, 0.00065, 0.00028, 0.00013, 0.00006,
     0.00005, 0.00001, 0.00002, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000},

    {1.00000, 1.00000, 1.00000, 1.00000, 0.99870, 0.98730, 0.93240, 0.81330, 0.62730, 0.45400,
     0.29000, 0.17620, 0.09950, 0.05670, 0.03350, 0.01640, 0.01120, 0.00520, 0.00320, 0.00160,
     0.00110, 0.00060, 0.00040, 0.00010, 0.00010, 0.00010, 0.00000, 0.00000, 0.00000},

    {0.00000, 0.00000, 0.00000, 0.00000, 0.00130, 0.01254, 0.06303, 0.15184, 0.23380, 0.24788,
     0.20590, 0.14515, 0.08960, 0.05348, 0.03238, 0.01613, 0.01107, 0.00517, 0.00319, 0.00160,
     0.00110, 0.00020, 0.00060, 0.00000, 0.00010, 0.00010, 0.00000, 0.00000, 0.00000},

    {1.00000, 1.00000, 1.00000, 1.00000, 0.99799, 0.98511, 0.92748, 0.80566, 0.61782, 0.44424,
     0.28111, 0.16873, 0.09363, 0.05217, 0.02997, 0.01391, 0.00914, 0.00379, 0.00209, 0.00082,
     0.00045, 0.00008, 0.00012, 0.00000, 0.00010, 0.00010, 0.00000, 0.00000, 0.00000},

    {1.00000, 1.00000, 1.00000, 1.00000, 0.99941, 0.98950, 0.93732, 0.82094, 0.63678, 0.46376,
     0.29889, 0.18367, 0.10537, 0.06123, 0.03703, 0.01889, 0.01326, 0.00661, 0.00431, 0.00238,
     0.00175, 0.00048, 0.00108, 0.00000, 0.00030, 0.00030, 0.00000, 0.00000, 0.00000},
};

static const double modulation1[6][42] = {

    {2.00, 2.10, 2.20, 2.30, 2.40, 2.50, 2.60, 2.70, 2.80, 2.90, 3.00, 3.10, 3.20, 3.30,
     3.40, 3.50, 3.60, 3.70, 3.80, 3.90, 4.00, 4.10, 4.20, 4.30, 4.40, 4.50, 4.60, 4.70,
     4.80, 4.90, 5.00, 5.10, 5.20, 5.30, 5.40, 5.50, 5.60, 5.70, 5.80, 5.90, 6.00, 6.10},

    {0.45604, 0.43627, 0.41185, 0.38360, 0.34393, 0.30548, 0.26033, 0.22148, 0.18646,
     0.14898, 0.11921, 0.09493, 0.07055, 0.05625, 0.04178, 0.03245, 0.02613, 0.01733,
     0.01415, 0.01151, 0.00834, 0.00506, 0.00450, 0.00349, 0.00213, 0.00158, 0.00162,
     0.00078, 0.00070, 0.00039, 0.00039, 0.00059, 0.00049, 0.00005, 0.00015, 0.00010,
     0.00000, 0.00029, 0.00005, 0.00000, 0.00005, 0.00000},

    {0.95150, 0.91120, 0.86240, 0.80300, 0.72060, 0.64010, 0.54740, 0.46440, 0.39190,
     0.31310, 0.25110, 0.19930, 0.14830, 0.11820, 0.08800, 0.06890, 0.05520, 0.03650,
     0.02980, 0.02440, 0.01770, 0.01090, 0.00920, 0.00730, 0.00460, 0.00340, 0.00340,
     0.00160, 0.00160, 0.00150, 0.00140, 0.00120, 0.00100, 0.00090, 0.00060, 0.00040,
     0.00020, 0.00010, 0.00005, 0.00004, 0.00003, 0.00000},

    {0.04615, 0.08092, 0.11867, 0.15819, 0.20134, 0.23037, 0.24775, 0.24873, 0.23831,
     0.21507, 0.18805, 0.15958, 0.12631, 0.10423, 0.08026, 0.06415, 0.05215, 0.03517,
     0.02891, 0.02380, 0.01739, 0.01078, 0.00911, 0.00725, 0.00458, 0.00339, 0.00339,
     0.00160, 0.00160, 0.00080, 0.00080, 0.00120, 0.00100, 0.00010, 0.00030, 0.00020,
     0.00000, 0.00060, 0.00010, 0.00000, 0.00010, 0.00000},

    {0.94729, 0.90563, 0.85565, 0.79520, 0.71181, 0.63069, 0.53764, 0.45463, 0.38233,
     0.30401, 0.24260, 0.19147, 0.14133, 0.11187, 0.08245, 0.06394, 0.05072, 0.03282,
     0.02647, 0.02138, 0.01512, 0.00886, 0.00733, 0.00563, 0.00327, 0.00226, 0.00226,
     0.00082, 0.00082, 0.00025, 0.00025, 0.00052, 0.00038, 0.00010, 0.00004, 0.00008,
     0.00000, 0.00012, 0.00010, 0.00000, 0.00010, 0.00000},

    {0.95571, 0.91678, 0.86915, 0.81080, 0.72940, 0.64951, 0.55716, 0.47418, 0.40147,
     0.32219, 0.25960, 0.20713, 0.15527, 0.12453, 0.09355, 0.07386, 0.05968, 0.04018,
     0.03313, 0.02742, 0.02028, 0.01294, 0.01107, 0.00897, 0.00593, 0.00454, 0.00454,
     0.00238, 0.00238, 0.00135, 0.00135, 0.00188, 0.00162, 0.00030, 0.00064, 0.00048,
     0.00000, 0.00108, 0.00030, 0.00000, 0.00030, 0.00000},
};

static const double modulation2[6][96] = {

    {3.00,  3.10,  3.20,  3.30,  3.40,  3.50,  3.60,  3.70,  3.80,  3.90,  4.00,  4.10,
     4.20,  4.30,  4.40,  4.50,  4.60,  4.70,  4.80,  4.90,  5.00,  5.10,  5.20,  5.30,
     5.40,  5.50,  5.60,  5.70,  5.80,  5.90,  6.00,  6.10,  6.20,  6.30,  6.40,  6.50,
     6.60,  6.70,  6.80,  6.90,  7.00,  7.10,  7.20,  7.30,  7.40,  7.50,  7.60,  7.70,
     7.80,  7.90,  8.00,  8.10,  8.20,  8.30,  8.40,  8.50,  8.60,  8.70,  8.80,  8.90,
     9.00,  9.10,  9.20,  9.30,  9.40,  9.50,  9.60,  9.70,  9.80,  9.90,  10.00, 10.10,
     10.20, 10.30, 10.40, 10.50, 10.60, 10.70, 10.80, 10.90, 11.00, 11.10, 11.20, 11.30,
     11.40, 11.50, 11.60, 11.70, 11.80, 11.90, 12.00, 12.10, 12.20, 12.30, 12.40, 12.50},

    {0.36630, 0.35317, 0.34175, 0.33351, 0.31776, 0.30292, 0.28624, 0.27526, 0.26174, 0.24856,
     0.23007, 0.21865, 0.19945, 0.18414, 0.16967, 0.15747, 0.14530, 0.13298, 0.11771, 0.11115,
     0.09713, 0.09123, 0.08231, 0.07267, 0.06616, 0.05673, 0.05156, 0.04577, 0.04282, 0.03866,
     0.03513, 0.02856, 0.02680, 0.02406, 0.01948, 0.01927, 0.01580, 0.01473, 0.01292, 0.01266,
     0.01108, 0.00937, 0.00735, 0.00675, 0.00582, 0.00612, 0.00496, 0.00471, 0.00386, 0.00412,
     0.00333, 0.00387, 0.00234, 0.00257, 0.00204, 0.00161, 0.00179, 0.00128, 0.00114, 0.00136,
     0.00102, 0.00133, 0.00087, 0.00066, 0.00075, 0.00066, 0.00069, 0.00041, 0.00051, 0.00035,
     0.00033, 0.00034, 0.00025, 0.00033, 0.00038, 0.00036, 0.00012, 0.00005, 0.00024, 0.00015,
     0.00010, 0.00000, 0.00011, 0.00001, 0.00024, 0.00001, 0.00001, 0.00005, 0.00001, 0.00005,
     0.00010, 0.00005, 0.00009, 0.00000, 0.00000, 0.00000},

    {0.99130, 0.98790, 0.97490, 0.96360, 0.94840, 0.92520, 0.89860, 0.87960, 0.84970, 0.81960,
     0.76860, 0.72540, 0.69130, 0.63620, 0.58540, 0.54710, 0.51010, 0.47450, 0.42570, 0.39700,
     0.34730, 0.32840, 0.29360, 0.25990, 0.23970, 0.20910, 0.18810, 0.16480, 0.15410, 0.14130,
     0.12570, 0.10670, 0.09940, 0.08810, 0.07330, 0.06930, 0.05720, 0.05530, 0.04630, 0.04460,
     0.04070, 0.03310, 0.02810, 0.02680, 0.02290, 0.02260, 0.01700, 0.01740, 0.01320, 0.01490,
     0.01210, 0.01320, 0.00820, 0.00850, 0.00840, 0.00640, 0.00590, 0.00480, 0.00450, 0.00490,
     0.00410, 0.00420, 0.00340, 0.00250, 0.00240, 0.00240, 0.00210, 0.00190, 0.00160, 0.00140,
     0.00120, 0.00110, 0.00100, 0.00100, 0.00100, 0.00100, 0.00080, 0.00070, 0.00050, 0.00050,
     0.00040, 0.00040, 0.00030, 0.00030, 0.00020, 0.00020, 0.00010, 0.00008, 0.00007, 0.00005,
     0.00004, 0.00003, 0.00003, 0.00003, 0.00002, 0.00000},

    {0.00862, 0.01195, 0.02447, 0.03508, 0.04894, 0.06921, 0.09112, 0.10590, 0.12771, 0.14786,
     0.17785, 0.19920, 0.21340, 0.23145, 0.24271, 0.24778, 0.24990, 0.24935, 0.24448, 0.23939,
     0.22668, 0.22055, 0.20740, 0.19235, 0.18224, 0.16538, 0.15272, 0.13764, 0.13035, 0.12133,
     0.10990, 0.09531, 0.08952, 0.08034, 0.06793, 0.06450, 0.05393, 0.05224, 0.04416, 0.04261,
     0.03904, 0.03200, 0.02731, 0.02608, 0.02238, 0.02209, 0.01671, 0.01710, 0.01303, 0.01468,
     0.01195, 0.01303, 0.00813, 0.00843, 0.00833, 0.00636, 0.00587, 0.00478, 0.00448, 0.00488,
     0.00408, 0.00418, 0.00339, 0.00230, 0.00239, 0.00239, 0.00210, 0.00110, 0.00190, 0.00120,
     0.00160, 0.00100, 0.00110, 0.00150, 0.00100, 0.00120, 0.00070, 0.00020, 0.00080, 0.00050,
     0.00030, 0.00000, 0.00050, 0.00010, 0.00050, 0.00010, 0.00010, 0.00010, 0.00020, 0.00020,
     0.00030, 0.00010, 0.00020, 0.00000, 0.00000, 0.00000},

    {0.98948, 0.98576, 0.97183, 0.95993, 0.94406, 0.92004, 0.89268, 0.87322, 0.84270, 0.81206,
     0.76033, 0.71665, 0.68225, 0.62677, 0.57574, 0.53734, 0.50030, 0.46471, 0.41601, 0.38741,
     0.33797, 0.31920, 0.28467, 0.25130, 0.23133, 0.20113, 0.18044, 0.15753, 0.14702, 0.13447,
     0.11920, 0.10065, 0.09354, 0.08254, 0.06819, 0.06432, 0.05265, 0.05082, 0.04218, 0.04055,
     0.03683, 0.02959, 0.02486, 0.02363, 0.01997, 0.01969, 0.01447, 0.01484, 0.01096, 0.01252,
     0.00996, 0.01096, 0.00643, 0.00670, 0.00661, 0.00484, 0.00440, 0.00345, 0.00319, 0.00353,
     0.00285, 0.00293, 0.00226, 0.00136, 0.00144, 0.00144, 0.00120, 0.00045, 0.00105, 0.00052,
     0.00082, 0.00038, 0.00045, 0.00074, 0.00038, 0.00052, 0.00018, 0.00008, 0.00025, 0.00006,
     0.00004, 0.00000, 0.00006, 0.00010, 0.00006, 0.00010, 0.00010, 0.00010, 0.00008, 0.00008,
     0.00004, 0.00010, 0.00008, 0.00000, 0.00000, 0.00000},

    {0.99312, 0.99004, 0.97797, 0.96727, 0.95274, 0.93036, 0.90452, 0.88598, 0.85670, 0.82714,
     0.77687, 0.73415, 0.70035, 0.64563, 0.59506, 0.55686, 0.51990, 0.48429, 0.43539, 0.40659,
     0.35663, 0.33760, 0.30253, 0.26850, 0.24807, 0.21707, 0.19576, 0.17207, 0.16118, 0.14813,
     0.13220, 0.11275, 0.10526, 0.09366, 0.07841, 0.07428, 0.06175, 0.05978, 0.05042, 0.04865,
     0.04457, 0.03661, 0.03134, 0.02996, 0.02583, 0.02551, 0.01953, 0.01996, 0.01544, 0.01727,
     0.01424, 0.01544, 0.00997, 0.01030, 0.01019, 0.00796, 0.00740, 0.00615, 0.00581, 0.00627,
     0.00535, 0.00547, 0.00454, 0.00324, 0.00336, 0.00336, 0.00300, 0.00175, 0.00275, 0.00188,
     0.00238, 0.00162, 0.00175, 0.00226, 0.00162, 0.00188, 0.00122, 0.00048, 0.00135, 0.00094,
     0.00064, 0.00000, 0.00094, 0.00030, 0.00094, 0.00030, 0.00030, 0.00030, 0.00048, 0.00048,
     0.00064, 0.00030, 0.00048, 0.00000, 0.00000, 0.00000},
};

static const double modulation3[6][117] = {

    {10.00, 10.10, 10.20, 10.30, 10.40, 10.50, 10.60, 10.70, 10.80, 10.90, 11.00, 11.10, 11.20,
     11.30, 11.40, 11.50, 11.60, 11.70, 11.80, 11.90, 12.00, 12.10, 12.20, 12.30, 12.40, 12.50,
     12.60, 12.70, 12.80, 12.90, 13.00, 13.10, 13.20, 13.30, 13.40, 13.50, 13.60, 13.70, 13.80,
     13.90, 14.00, 14.10, 14.20, 14.30, 14.40, 14.50, 14.60, 14.70, 14.90, 15.00, 15.10, 15.20,
     15.30, 15.40, 15.50, 15.60, 15.70, 15.80, 15.90, 16.00, 16.10, 16.20, 16.30, 16.40, 16.50,
     16.60, 16.70, 16.80, 16.90, 17.10, 17.20, 17.30, 17.40, 17.50, 17.60, 17.70, 17.80, 17.90,
     18.00, 18.10, 18.20, 18.30, 18.40, 18.50, 18.60, 18.70, 18.80, 18.90, 19.00, 19.10, 19.20,
     19.30, 19.40, 19.50, 19.60, 19.70, 19.80, 19.90, 20.00, 20.10, 20.20, 20.30, 20.40, 20.50,
     20.60, 20.70, 20.80, 20.90, 21.00, 21.10, 21.20, 21.30, 21.40, 21.50, 21.60, 21.70, 21.80},

    {0.39109, 0.37994, 0.36904, 0.36186, 0.35172, 0.34069, 0.32445, 0.31707, 0.30757, 0.28951,
     0.27937, 0.26696, 0.25737, 0.24826, 0.23413, 0.22045, 0.21144, 0.20244, 0.19049, 0.17783,
     0.16793, 0.15746, 0.14697, 0.13387, 0.12757, 0.12354, 0.11370, 0.10474, 0.09701, 0.09219,
     0.08535, 0.07516, 0.07240, 0.06492, 0.06394, 0.05577, 0.05399, 0.05040, 0.04632, 0.04287,
     0.03918, 0.03327, 0.03376, 0.02855, 0.02791, 0.02417, 0.02412, 0.02111, 0.01801, 0.01482,
     0.01486, 0.01324, 0.01167, 0.01137, 0.01127, 0.00911, 0.00930, 0.00650, 0.00748, 0.00615,
     0.00610, 0.00487, 0.00492, 0.00364, 0.00448, 0.00477, 0.00231, 0.00295, 0.00256, 0.00266,
     0.00182, 0.00192, 0.00182, 0.00123, 0.00158, 0.00153, 0.00153, 0.00093, 0.00093, 0.00064,
     0.00064, 0.00059, 0.00069, 0.00034, 0.00034, 0.00054, 0.00039, 0.00010, 0.00025, 0.00049,
     0.00030, 0.00034, 0.00015, 0.00034, 0.00030, 0.00034, 0.00005, 0.00020, 0.00020, 0.00015,
     0.00010, 0.00010, 0.00010, 0.00010, 0.00010, 0.00020, 0.00000, 0.00005, 0.00000, 0.00005,
     0.00010, 0.00020, 0.00010, 0.00005, 0.00005, 0.00020, 0.00000},

    {0.79460, 0.77200, 0.74980, 0.73520, 0.71470, 0.69220, 0.65920, 0.64420, 0.62490, 0.58830,
     0.56760, 0.54240, 0.52290, 0.50440, 0.47570, 0.44790, 0.42960, 0.41130, 0.38710, 0.36140,
     0.34120, 0.32000, 0.29860, 0.27200, 0.25920, 0.25100, 0.23100, 0.21280, 0.19710, 0.18730,
     0.17340, 0.15270, 0.14710, 0.13190, 0.12990, 0.11330, 0.10970, 0.10240, 0.09410, 0.08710,
     0.07960, 0.06760, 0.06860, 0.05800, 0.05670, 0.04910, 0.04900, 0.04290, 0.03660, 0.03010,
     0.03020, 0.02690, 0.02370, 0.02310, 0.02290, 0.01850, 0.01890, 0.01520, 0.01320, 0.01250,
     0.01240, 0.01000, 0.01000, 0.00940, 0.00910, 0.00970, 0.00700, 0.00600, 0.00520, 0.00540,
     0.00390, 0.00390, 0.00370, 0.00320, 0.00320, 0.00310, 0.00310, 0.00190, 0.00190, 0.00130,
     0.00130, 0.00120, 0.00140, 0.00090, 0.00080, 0.00070, 0.00070, 0.00070, 0.00060, 0.00060,
     0.00060, 0.00050, 0.00050, 0.00050, 0.00040, 0.00040, 0.00040, 0.00040, 0.00040, 0.00030,
     0.00020, 0.00020, 0.00020, 0.00020, 0.00020, 0.00015, 0.00014, 0.00013, 0.00012, 0.00011,
     0.00010, 0.00009, 0.00008, 0.00007, 0.00006, 0.00005, 0.00000},

    {0.16321, 0.17602, 0.18760, 0.19468, 0.20390, 0.21306, 0.22465, 0.22921, 0.23440, 0.24220,
     0.24543, 0.24820, 0.24948, 0.24998, 0.24941, 0.24729, 0.24504, 0.24213, 0.23725, 0.23079,
     0.22478, 0.21760, 0.20944, 0.19802, 0.19201, 0.18800, 0.17764, 0.16752, 0.15825, 0.15222,
     0.14333, 0.12938, 0.12546, 0.11450, 0.11303, 0.10046, 0.09767, 0.09191, 0.08525, 0.07951,
     0.07326, 0.06303, 0.06389, 0.05464, 0.05348, 0.04669, 0.04660, 0.04106, 0.03526, 0.02919,
     0.02929, 0.02618, 0.02314, 0.02257, 0.02238, 0.01816, 0.01854, 0.01303, 0.01497, 0.01234,
     0.01225, 0.00980, 0.00990, 0.00734, 0.00902, 0.00961, 0.00468, 0.00596, 0.00517, 0.00537,
     0.00369, 0.00388, 0.00369, 0.00249, 0.00319, 0.00309, 0.00309, 0.00190, 0.00190, 0.00130,
     0.00130, 0.00120, 0.00140, 0.00070, 0.00070, 0.00110, 0.00080, 0.00020, 0.00050, 0.00100,
     0.00060, 0.00070, 0.00030, 0.00070, 0.00060, 0.00070, 0.00010, 0.00040, 0.00040, 0.00030,
     0.00020, 0.00020, 0.00020, 0.00020, 0.00020, 0.00040, 0.00000, 0.00010, 0.00000, 0.00010,
     0.00020, 0.00040, 0.00020, 0.00010, 0.00010, 0.00040, 0.00000},

    {0.78668, 0.76378, 0.74131, 0.72655, 0.70585, 0.68315, 0.64991, 0.63482, 0.61541, 0.57865,
     0.55789, 0.53263, 0.51311, 0.49460, 0.46591, 0.43815, 0.41990, 0.40165, 0.37755, 0.35198,
     0.33191, 0.31086, 0.28963, 0.26328, 0.25061, 0.24250, 0.22274, 0.20478, 0.18930, 0.17965,
     0.16598, 0.14565, 0.14016, 0.12527, 0.12331, 0.10709, 0.10358, 0.09646, 0.08838, 0.08157,
     0.07429, 0.06268, 0.06365, 0.05342, 0.05217, 0.04487, 0.04477, 0.03893, 0.03292, 0.02675,
     0.02685, 0.02373, 0.02072, 0.02016, 0.01997, 0.01586, 0.01623, 0.01096, 0.01280, 0.01032,
     0.01023, 0.00796, 0.00805, 0.00572, 0.00724, 0.00778, 0.00336, 0.00449, 0.00379, 0.00396,
     0.00251, 0.00268, 0.00251, 0.00152, 0.00209, 0.00201, 0.00201, 0.00105, 0.00105, 0.00059,
     0.00059, 0.00052, 0.00067, 0.00018, 0.00018, 0.00045, 0.00025, 0.00008, 0.00006, 0.00038,
     0.00012, 0.00018, 0.00004, 0.00018, 0.00012, 0.00018, 0.00010, 0.00001, 0.00001, 0.00004,
     0.00008, 0.00008, 0.00008, 0.00008, 0.00008, 0.00001, 0.00000, 0.00010, 0.00000, 0.00010,
     0.00008, 0.00001, 0.00008, 0.00010, 0.00010, 0.00001, 0.00000},

    {0.80252, 0.78022, 0.75829, 0.74385, 0.72355, 0.70125, 0.66849, 0.65358, 0.63439, 0.59795,
     0.57731, 0.55217, 0.53269, 0.51420, 0.48549, 0.45765, 0.43930, 0.42095, 0.39665, 0.37082,
     0.35049, 0.32914, 0.30757, 0.28072, 0.26779, 0.25950, 0.23926, 0.22082, 0.20490, 0.19495,
     0.18082, 0.15975, 0.15404, 0.13853, 0.13649, 0.11951, 0.11582, 0.10834, 0.09982, 0.09263,
     0.08490, 0.07252, 0.07355, 0.06258, 0.06123, 0.05334, 0.05323, 0.04687, 0.04028, 0.03345,
     0.03355, 0.03007, 0.02668, 0.02604, 0.02583, 0.02114, 0.02157, 0.01544, 0.01760, 0.01468,
     0.01457, 0.01184, 0.01195, 0.00908, 0.01096, 0.01162, 0.00604, 0.00751, 0.00661, 0.00684,
     0.00489, 0.00512, 0.00489, 0.00348, 0.00431, 0.00419, 0.00419, 0.00275, 0.00275, 0.00201,
     0.00201, 0.00188, 0.00213, 0.00122, 0.00122, 0.00175, 0.00135, 0.00048, 0.00094, 0.00162,
     0.00108, 0.00122, 0.00064, 0.00122, 0.00108, 0.00122, 0.00030, 0.00079, 0.00079, 0.00064,
     0.00048, 0.00048, 0.00048, 0.00048, 0.00048, 0.00079, 0.00000, 0.00030, 0.00000, 0.00030,
     0.00048, 0.00079, 0.00048, 0.00030, 0.00030, 0.00079, 0.00000},
};

static const double modulation4[6][331] = {

    {20,   20.1, 20.2, 20.3, 20.4, 20.5, 20.6, 20.7, 20.8, 20.9, 21,   21.1, 21.2, 21.3, 21.4, 21.5,
     21.6, 21.7, 21.8, 21.9, 22,   22.1, 22.2, 22.3, 22.4, 22.5, 22.6, 22.7, 22.8, 22.9, 23,   23.1,
     23.2, 23.3, 23.4, 23.5, 23.6, 23.7, 23.8, 23.9, 24,   24.1, 24.2, 24.3, 24.4, 24.5, 24.6, 24.7,
     24.8, 24.9, 25,   25.1, 25.2, 25.3, 25.4, 25.5, 25.6, 25.7, 25.8, 25.9, 26,   26.1, 26.2, 26.3,
     26.4, 26.5, 26.6, 26.7, 26.8, 26.9, 27,   27.1, 27.2, 27.3, 27.4, 27.5, 27.6, 27.7, 27.8, 27.9,
     28,   28.1, 28.2, 28.3, 28.4, 28.5, 28.6, 28.7, 28.8, 28.9, 29,   29.1, 29.2, 29.3, 29.4, 29.5,
     29.6, 29.7, 29.8, 29.9, 30,   30.1, 30.2, 30.3, 30.4, 30.5, 30.6, 30.7, 30.8, 30.9, 31,   31.1,
     31.2, 31.3, 31.4, 31.5, 31.6, 31.7, 31.8, 31.9, 32,   32.1, 32.2, 32.3, 32.4, 32.5, 32.6, 32.7,
     32.8, 32.9, 33,   33.1, 33.2, 33.3, 33.4, 33.5, 33.6, 33.7, 33.8, 33.9, 34,   34.1, 34.2, 34.3,
     34.4, 34.5, 34.6, 34.7, 34.8, 34.9, 35,   35.1, 35.2, 35.3, 35.4, 35.5, 35.6, 35.7, 35.8, 35.9,
     36,   36.1, 36.2, 36.3, 36.4, 36.5, 36.6, 36.7, 36.8, 36.9, 37,   37.1, 37.2, 37.3, 37.4, 37.5,
     37.6, 37.7, 37.8, 37.9, 38,   38.1, 38.2, 38.3, 38.4, 38.5, 38.6, 38.7, 38.8, 38.9, 39,   39.1,
     39.2, 39.3, 39.4, 39.5, 39.6, 39.7, 39.8, 39.9, 40,   40.1, 40.2, 40.3, 40.4, 40.5, 40.6, 40.7,
     40.8, 40.9, 41,   41.1, 41.2, 41.3, 41.4, 41.5, 41.6, 41.7, 41.8, 41.9, 42,   42.1, 42.2, 42.3,
     42.4, 42.5, 42.6, 42.7, 42.8, 42.9, 43,   43.1, 43.2, 43.3, 43.4, 43.5, 43.6, 43.7, 43.8, 43.9,
     44,   44.1, 44.2, 44.3, 44.4, 44.5, 44.6, 44.7, 44.8, 44.9, 45,   45.1, 45.2, 45.3, 45.4, 45.5,
     45.6, 45.7, 45.8, 45.9, 46,   46.1, 46.2, 46.3, 46.4, 46.5, 46.6, 46.7, 46.8, 46.9, 47,   47.1,
     47.2, 47.3, 47.4, 47.5, 47.6, 47.7, 47.8, 47.9, 48,   48.1, 48.2, 48.3, 48.4, 48.5, 48.6, 48.7,
     48.8, 48.9, 49,   49.1, 49.2, 49.3, 49.4, 49.5, 49.6, 49.7, 49.8, 49.9, 50,   50.1, 50.2, 50.3,
     50.4, 50.5, 50.6, 50.7, 50.8, 50.9, 51,   51.1, 51.2, 51.3, 51.4, 51.5, 51.6, 51.7, 51.8, 51.9,
     52,   52.1, 52.2, 52.3, 52.4, 52.5, 52.6, 52.7, 52.8, 52.9, 53},

    {0.473383, 0.465027, 0.456673, 0.448325, 0.439984, 0.431653, 0.423335, 0.415032, 0.406746,
     0.398480, 0.390237, 0.382020, 0.373830, 0.365672, 0.357547, 0.349458, 0.341409, 0.333402,
     0.325439, 0.317525, 0.309662, 0.301852, 0.294099, 0.286406, 0.278776, 0.271211, 0.263714,
     0.256290, 0.248939, 0.241667, 0.234474, 0.227364, 0.220341, 0.213406, 0.206562, 0.199813,
     0.193161, 0.186608, 0.180156, 0.173809, 0.167569, 0.161438, 0.155418, 0.149511, 0.143719,
     0.138045, 0.132489, 0.127053, 0.121740, 0.116550, 0.111485, 0.106546, 0.101733, 0.097049,
     0.092492, 0.088064, 0.083766, 0.079597, 0.075558, 0.071648, 0.067867, 0.064215, 0.060691,
     0.057295, 0.054025, 0.050880, 0.047860, 0.044962, 0.042185, 0.039528, 0.036989, 0.034565,
     0.032255, 0.030056, 0.027965, 0.025981, 0.024101, 0.022322, 0.020641, 0.019055, 0.017561,
     0.016157, 0.014839, 0.013604, 0.012449, 0.011371, 0.010366, 0.009431, 0.008563, 0.007760,
     0.007017, 0.006331, 0.005700, 0.005120, 0.004589, 0.004103, 0.003660, 0.003256, 0.002890,
     0.002558, 0.002259, 0.001989, 0.001747, 0.001529, 0.001335, 0.001162, 0.001009, 0.000873,
     0.000753, 0.000647, 0.000555, 0.000474, 0.000403, 0.000342, 0.000289, 0.000243, 0.000204,
     0.000170, 0.000142, 0.000117, 0.000097, 0.000080, 0.000065, 0.000053, 0.000043, 0.000035,
     0.000028, 0.000022, 0.000018, 0.000014, 0.000011, 0.000009, 0.000007, 0.000005, 0.000004,
     0.000003, 0.000002, 0.000002, 0.000001, 0.000001, 0.000001, 0.000001, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000},

    {0.899428, 0.883551, 0.867679, 0.851817, 0.835970, 0.820141, 0.804337, 0.788561, 0.772818,
     0.757113, 0.741451, 0.725838, 0.710278, 0.694776, 0.679339, 0.663970, 0.648677, 0.633463,
     0.618335, 0.603298, 0.588357, 0.573519, 0.558789, 0.544172, 0.529674, 0.515300, 0.501058,
     0.486950, 0.472985, 0.459166, 0.445500, 0.431992, 0.418647, 0.405471, 0.392468, 0.379645,
     0.367005, 0.354554, 0.342297, 0.330238, 0.318382, 0.306732, 0.295294, 0.284071, 0.273066,
     0.262285, 0.251729, 0.241402, 0.231306, 0.221446, 0.211822, 0.202437, 0.193293, 0.184392,
     0.175735, 0.167322, 0.159155, 0.151234, 0.143559, 0.136131, 0.128947, 0.122008, 0.115313,
     0.108860, 0.102647, 0.096672, 0.090933, 0.085428, 0.080152, 0.075104, 0.070279, 0.065674,
     0.061284, 0.057106, 0.053134, 0.049364, 0.045792, 0.042411, 0.039217, 0.036204, 0.033367,
     0.030699, 0.028194, 0.025848, 0.023653, 0.021604, 0.019695, 0.017919, 0.016271, 0.014743,
     0.013332, 0.012029, 0.010830, 0.009728, 0.008719, 0.007796, 0.006953, 0.006187, 0.005491,
     0.004861, 0.004292, 0.003779, 0.003318, 0.002906, 0.002537, 0.002209, 0.001917, 0.001659,
     0.001430, 0.001230, 0.001054, 0.000900, 0.000766, 0.000649, 0.000549, 0.000462, 0.000387,
     0.000324, 0.000269, 0.000223, 0.000184, 0.000151, 0.000124, 0.000101, 0.000082, 0.000066,
     0.000053, 0.000042, 0.000034, 0.000027, 0.000021, 0.000016, 0.000013, 0.000010, 0.000008,
     0.000006, 0.000004, 0.000003, 0.000003, 0.000002, 0.000001, 0.000001, 0.000001, 0.000001,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000},

    {0.163210, 0.176020, 0.187600, 0.194680, 0.203900, 0.213060, 0.224650, 0.229210, 0.234400,
     0.242200, 0.245430, 0.248200, 0.249480, 0.249980, 0.249410, 0.247290, 0.245040, 0.242130,
     0.237250, 0.230790, 0.224780, 0.217600, 0.209440, 0.198020, 0.192010, 0.188000, 0.177640,
     0.167520, 0.158250, 0.152220, 0.143330, 0.129380, 0.125460, 0.114500, 0.113030, 0.100460,
     0.097670, 0.091910, 0.085250, 0.079510, 0.073260, 0.063030, 0.063890, 0.054640, 0.053480,
     0.046690, 0.046600, 0.041060, 0.035260, 0.029190, 0.029290, 0.026180, 0.023140, 0.022570,
     0.022380, 0.018160, 0.018540, 0.013030, 0.014970, 0.012340, 0.012250, 0.009800, 0.009900,
     0.007340, 0.009020, 0.009610, 0.004680, 0.005960, 0.005170, 0.005370, 0.003690, 0.003880,
     0.003690, 0.002490, 0.003190, 0.003090, 0.003090, 0.001900, 0.001900, 0.001300, 0.001300,
     0.001200, 0.001400, 0.000700, 0.000700, 0.001100, 0.000800, 0.000200, 0.000500, 0.001000,
     0.000600, 0.000700, 0.000300, 0.000700, 0.000600, 0.000700, 0.000100, 0.000400, 0.000400,
     0.000300, 0.000200, 0.000200, 0.000200, 0.000200, 0.000200, 0.000400, 0.000000, 0.000100,
     0.000000, 0.000100, 0.000200, 0.000400, 0.000200, 0.000100, 0.000100, 0.000400, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000},

    {0.890434, 0.874715, 0.859002, 0.843299, 0.827610, 0.811940, 0.796293, 0.780675, 0.765089,
     0.749542, 0.734037, 0.718579, 0.703175, 0.687828, 0.672545, 0.657331, 0.642190, 0.627128,
     0.612151, 0.597265, 0.582474, 0.567784, 0.553201, 0.538730, 0.524377, 0.510147, 0.496047,
     0.482081, 0.468255, 0.454575, 0.441045, 0.427672, 0.414461, 0.401416, 0.388544, 0.375848,
     0.363335, 0.351009, 0.338874, 0.326936, 0.315198, 0.303665, 0.292341, 0.281230, 0.270336,
     0.259662, 0.249211, 0.238988, 0.228993, 0.219231, 0.209704, 0.200413, 0.191361, 0.182548,
     0.173977, 0.165649, 0.157564, 0.149722, 0.142124, 0.134769, 0.127658, 0.120788, 0.114160,
     0.107771, 0.101620, 0.095705, 0.090024, 0.084573, 0.079351, 0.074353, 0.069576, 0.065017,
     0.060671, 0.056535, 0.052603, 0.048871, 0.045334, 0.041987, 0.038825, 0.035842, 0.033033,
     0.030392, 0.027912, 0.025589, 0.023417, 0.021388, 0.019498, 0.017740, 0.016108, 0.014596,
     0.013198, 0.011909, 0.010722, 0.009631, 0.008632, 0.007718, 0.006884, 0.006125, 0.005436,
     0.004812, 0.004249, 0.003741, 0.003285, 0.002877, 0.002512, 0.002187, 0.001898, 0.001642,
     0.001416, 0.001218, 0.001043, 0.000891, 0.000758, 0.000643, 0.000543, 0.000457, 0.000384,
     0.000320, 0.000267, 0.000221, 0.000182, 0.000150, 0.000123, 0.000100, 0.000081, 0.000065,
     0.000052, 0.000042, 0.000033, 0.000026, 0.000021, 0.000016, 0.000013, 0.000010, 0.000008,
     0.000006, 0.000004, 0.000003, 0.000003, 0.000002, 0.000001, 0.000001, 0.000001, 0.000001,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000},

    {0.908423, 0.892386, 0.876356, 0.860335, 0.844330, 0.828343, 0.812380, 0.796446, 0.780546,
     0.764684, 0.748866, 0.733096, 0.717380, 0.701724, 0.686132, 0.670610, 0.655163, 0.639798,
     0.624518, 0.609331, 0.594241, 0.579254, 0.564377, 0.549613, 0.534970, 0.520453, 0.506068,
     0.491820, 0.477715, 0.463758, 0.449955, 0.436312, 0.422834, 0.409526, 0.396393, 0.383441,
     0.370675, 0.358100, 0.345720, 0.333540, 0.321565, 0.309799, 0.298247, 0.286911, 0.275797,
     0.264907, 0.254246, 0.243816, 0.233619, 0.223660, 0.213940, 0.204462, 0.195226, 0.186236,
     0.177492, 0.168995, 0.160747, 0.152747, 0.144995, 0.137492, 0.130237, 0.123228, 0.116466,
     0.109948, 0.103673, 0.097639, 0.091843, 0.086282, 0.080954, 0.075855, 0.070982, 0.066330,
     0.061897, 0.057677, 0.053665, 0.049858, 0.046250, 0.042835, 0.039609, 0.036566, 0.033700,
     0.031006, 0.028476, 0.026106, 0.023890, 0.021820, 0.019892, 0.018098, 0.016433, 0.014891,
     0.013465, 0.012149, 0.010938, 0.009826, 0.008806, 0.007873, 0.007023, 0.006249, 0.005546,
     0.004910, 0.004335, 0.003817, 0.003352, 0.002935, 0.002562, 0.002231, 0.001936, 0.001675,
     0.001445, 0.001242, 0.001064, 0.000909, 0.000774, 0.000656, 0.000554, 0.000467, 0.000391,
     0.000327, 0.000272, 0.000225, 0.000186, 0.000153, 0.000125, 0.000102, 0.000083, 0.000067,
     0.000054, 0.000043, 0.000034, 0.000027, 0.000021, 0.000017, 0.000013, 0.000010, 0.000008,
     0.000006, 0.000004, 0.000003, 0.000003, 0.000002, 0.000001, 0.000001, 0.000001, 0.000001,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
     0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000},
};

static const double modulation5[6][547] = {

    {50.00,  50.10,  50.20,  50.30,  50.40,  50.50,  50.60,  50.70,  50.80,  50.90,  51.00,  51.10,
     51.20,  51.30,  51.40,  51.50,  51.60,  51.70,  51.80,  51.90,  52.00,  52.10,  52.20,  52.30,
     52.40,  52.50,  52.60,  52.70,  52.80,  52.90,  53.00,  53.10,  53.20,  53.30,  53.40,  53.50,
     53.60,  53.70,  53.80,  53.90,  54.00,  54.10,  54.20,  54.30,  54.40,  54.50,  54.60,  54.70,
     54.80,  54.90,  55.00,  55.10,  55.20,  55.30,  55.40,  55.50,  55.60,  55.70,  55.80,  55.90,
     56.00,  56.10,  56.20,  56.30,  56.40,  56.50,  56.60,  56.70,  56.80,  56.90,  57.00,  57.10,
     57.20,  57.30,  57.40,  57.50,  57.60,  57.70,  57.80,  57.90,  58.00,  58.10,  58.20,  58.30,
     58.40,  58.50,  58.60,  58.70,  58.80,  58.90,  59.00,  59.10,  59.20,  59.30,  59.40,  59.50,
     59.60,  59.70,  59.80,  59.90,  60.00,  60.10,  60.20,  60.30,  60.40,  60.50,  60.60,  60.70,
     60.80,  60.90,  61.00,  61.10,  61.20,  61.30,  61.40,  61.50,  61.60,  61.70,  61.80,  61.90,
     62.00,  62.10,  62.20,  62.30,  62.40,  62.50,  62.60,  62.70,  62.80,  62.90,  63.00,  63.10,
     63.20,  63.30,  63.40,  63.50,  63.60,  63.70,  63.80,  63.90,  64.00,  64.10,  64.20,  64.30,
     64.40,  64.50,  64.60,  64.70,  64.80,  64.90,  65.00,  65.10,  65.20,  65.30,  65.40,  65.50,
     65.60,  65.70,  65.80,  65.90,  66.00,  66.10,  66.20,  66.30,  66.40,  66.50,  66.60,  66.70,
     66.80,  66.90,  67.00,  67.10,  67.20,  67.30,  67.40,  67.50,  67.60,  67.70,  67.80,  67.90,
     68.00,  68.10,  68.20,  68.30,  68.40,  68.50,  68.60,  68.70,  68.80,  68.90,  69.00,  69.10,
     69.20,  69.30,  69.40,  69.50,  69.60,  69.70,  69.80,  69.90,  70.00,  70.10,  70.20,  70.30,
     70.40,  70.50,  70.60,  70.70,  70.80,  70.90,  71.00,  71.10,  71.20,  71.30,  71.40,  71.50,
     71.60,  71.70,  71.80,  71.90,  72.00,  72.10,  72.20,  72.30,  72.40,  72.50,  72.60,  72.70,
     72.80,  72.90,  73.00,  73.10,  73.20,  73.30,  73.40,  73.50,  73.60,  73.70,  73.80,  73.90,
     74.00,  74.10,  74.20,  74.30,  74.40,  74.50,  74.60,  74.70,  74.80,  74.90,  75.00,  75.10,
     75.20,  75.30,  75.40,  75.50,  75.60,  75.70,  75.80,  75.90,  76.00,  76.10,  76.20,  76.30,
     76.40,  76.50,  76.60,  76.70,  76.80,  76.90,  77.00,  77.10,  77.20,  77.30,  77.40,  77.50,
     77.60,  77.70,  77.80,  77.90,  78.00,  78.10,  78.20,  78.30,  78.40,  78.50,  78.60,  78.70,
     78.80,  78.90,  79.00,  79.10,  79.20,  79.30,  79.40,  79.50,  79.60,  79.70,  79.80,  79.90,
     80.00,  80.10,  80.20,  80.30,  80.40,  80.50,  80.60,  80.70,  80.80,  80.90,  81.00,  81.10,
     81.20,  81.30,  81.40,  81.50,  81.60,  81.70,  81.80,  81.90,  82.00,  82.10,  82.20,  82.30,
     82.40,  82.50,  82.60,  82.70,  82.80,  82.90,  83.00,  83.10,  83.20,  83.30,  83.40,  83.50,
     83.60,  83.70,  83.80,  83.90,  84.00,  84.10,  84.20,  84.30,  84.40,  84.50,  84.60,  84.70,
     84.80,  84.90,  85.00,  85.10,  85.20,  85.30,  85.40,  85.50,  85.60,  85.70,  85.80,  85.90,
     86.00,  86.10,  86.20,  86.30,  86.40,  86.50,  86.60,  86.70,  86.80,  86.90,  87.00,  87.10,
     87.20,  87.30,  87.40,  87.50,  87.60,  87.70,  87.80,  87.90,  88.00,  88.10,  88.20,  88.30,
     88.40,  88.50,  88.60,  88.70,  88.80,  88.90,  89.00,  89.10,  89.20,  89.30,  89.40,  89.50,
     89.60,  89.70,  89.80,  89.90,  90.00,  90.10,  90.20,  90.30,  90.40,  90.50,  90.60,  90.70,
     90.80,  90.90,  91.00,  91.10,  91.20,  91.30,  91.40,  91.50,  91.60,  91.70,  91.80,  91.90,
     92.00,  92.10,  92.20,  92.30,  92.40,  92.50,  92.60,  92.70,  92.80,  92.90,  93.00,  93.10,
     93.20,  93.30,  93.40,  93.50,  93.60,  93.70,  93.80,  93.90,  94.00,  94.10,  94.20,  94.30,
     94.40,  94.50,  94.60,  94.70,  94.80,  94.90,  95.00,  96.00,  96.10,  96.20,  96.30,  96.40,
     96.50,  96.60,  96.70,  96.80,  96.90,  97.00,  97.10,  97.20,  97.30,  97.40,  97.50,  97.60,
     97.70,  97.80,  97.90,  98.00,  98.10,  98.20,  98.30,  98.40,  98.50,  98.60,  98.70,  98.80,
     98.90,  99.00,  99.10,  99.20,  99.30,  99.40,  99.50,  99.60,  99.70,  99.80,  99.90,  100.00,
     100.10, 100.20, 100.30, 100.40, 100.50, 100.60, 100.70, 100.80, 100.90, 101.00, 101.10, 101.20,
     101.30, 101.40, 101.50, 101.60, 101.70, 101.80, 101.90, 102.00, 102.10, 102.20, 102.30, 102.40,
     102.50, 102.60, 102.70, 102.80, 102.90, 103.00, 103.10, 103.20, 103.30, 103.40, 103.50, 103.60,
     103.70, 103.80, 103.90, 104.00, 104.10, 104.20, 104.30, 104.40, 104.50, 104.60, 104.70, 104.80,
     104.90, 105.00},

    {0.46874, 0.46753, 0.46863, 0.46778, 0.46771, 0.46740, 0.46399, 0.46725, 0.46422, 0.46493,
     0.46519, 0.46289, 0.46231, 0.46265, 0.46173, 0.46049, 0.45981, 0.45721, 0.45743, 0.45867,
     0.45599, 0.45719, 0.45515, 0.45272, 0.45396, 0.45432, 0.45263, 0.44926, 0.44898, 0.45062,
     0.45024, 0.44840, 0.44469, 0.44360, 0.44467, 0.44375, 0.44466, 0.44159, 0.44011, 0.43992,
     0.43979, 0.43863, 0.44184, 0.43579, 0.43560, 0.43449, 0.43185, 0.43021, 0.43098, 0.42802,
     0.42823, 0.42574, 0.42867, 0.42309, 0.42344, 0.42089, 0.42158, 0.41947, 0.41761, 0.41779,
     0.41695, 0.41723, 0.41540, 0.40973, 0.41315, 0.40834, 0.40579, 0.40460, 0.40443, 0.40311,
     0.39945, 0.40115, 0.40351, 0.39952, 0.39711, 0.39469, 0.39120, 0.39256, 0.39325, 0.39029,
     0.38713, 0.38405, 0.38310, 0.38170, 0.38153, 0.38150, 0.37932, 0.37625, 0.37639, 0.37513,
     0.36934, 0.37158, 0.36743, 0.36725, 0.36754, 0.36367, 0.36349, 0.36127, 0.35845, 0.35664,
     0.35571, 0.34821, 0.35324, 0.34971, 0.34971, 0.35046, 0.34550, 0.34232, 0.34107, 0.33753,
     0.34123, 0.33774, 0.33267, 0.33334, 0.32925, 0.32793, 0.32465, 0.32587, 0.32283, 0.32465,
     0.31899, 0.31939, 0.31609, 0.31355, 0.31146, 0.30996, 0.30590, 0.30941, 0.30495, 0.30232,
     0.30000, 0.30592, 0.30067, 0.29949, 0.29367, 0.29305, 0.29100, 0.28859, 0.28693, 0.28460,
     0.28802, 0.28214, 0.28012, 0.27655, 0.27398, 0.27489, 0.27265, 0.27207, 0.27215, 0.26801,
     0.27007, 0.26242, 0.26167, 0.25807, 0.26060, 0.25424, 0.25617, 0.25184, 0.25211, 0.24963,
     0.24654, 0.24889, 0.24176, 0.24156, 0.24427, 0.23891, 0.23345, 0.23369, 0.23111, 0.22584,
     0.22942, 0.22945, 0.22418, 0.22058, 0.22445, 0.21787, 0.22024, 0.22024, 0.21310, 0.21312,
     0.21306, 0.21014, 0.21204, 0.20921, 0.20454, 0.20235, 0.20241, 0.19851, 0.20200, 0.19926,
     0.19586, 0.19828, 0.19279, 0.19168, 0.18899, 0.18156, 0.18563, 0.18633, 0.18394, 0.18089,
     0.17964, 0.17685, 0.17630, 0.17696, 0.17502, 0.17138, 0.17179, 0.17177, 0.16918, 0.16656,
     0.16816, 0.16230, 0.16106, 0.15727, 0.15690, 0.15554, 0.15106, 0.15557, 0.14960, 0.15174,
     0.14814, 0.14777, 0.14698, 0.14451, 0.14344, 0.14489, 0.14124, 0.14074, 0.13459, 0.13994,
     0.13606, 0.13941, 0.13231, 0.13203, 0.12701, 0.13091, 0.12635, 0.12740, 0.12701, 0.12425,
     0.12362, 0.12399, 0.12174, 0.12099, 0.11940, 0.11865, 0.11551, 0.11818, 0.11366, 0.11303,
     0.11050, 0.10948, 0.11105, 0.10813, 0.11124, 0.10642, 0.10827, 0.11041, 0.10285, 0.10473,
     0.09880, 0.10248, 0.09540, 0.09905, 0.09991, 0.09618, 0.09506, 0.09641, 0.09405, 0.09332,
     0.09131, 0.09068, 0.09243, 0.08471, 0.08641, 0.08408, 0.08723, 0.08609, 0.08705, 0.08511,
     0.08122, 0.08053, 0.07947, 0.07747, 0.08165, 0.07549, 0.07670, 0.07824, 0.07534, 0.07525,
     0.07588, 0.07390, 0.07203, 0.07413, 0.07441, 0.06786, 0.07505, 0.06733, 0.06543, 0.06713,
     0.06481, 0.06620, 0.06451, 0.06596, 0.06732, 0.06286, 0.06340, 0.06110, 0.06586, 0.05799,
     0.06174, 0.05731, 0.06178, 0.05907, 0.05567, 0.05586, 0.05653, 0.05728, 0.05270, 0.05378,
     0.05503, 0.05143, 0.05158, 0.05056, 0.04994, 0.04988, 0.04934, 0.05066, 0.04965, 0.04614,
     0.04920, 0.04983, 0.04556, 0.04978, 0.04673, 0.04594, 0.04701, 0.04590, 0.04294, 0.04526,
     0.04284, 0.04478, 0.04237, 0.03920, 0.03740, 0.03988, 0.04055, 0.03832, 0.03959, 0.03906,
     0.03856, 0.03672, 0.03585, 0.03492, 0.03618, 0.03672, 0.03633, 0.03628, 0.03366, 0.03619,
     0.03560, 0.03449, 0.03371, 0.03517, 0.03176, 0.03113, 0.03128, 0.03226, 0.03109, 0.03163,
     0.02916, 0.02963, 0.03264, 0.03274, 0.02953, 0.02802, 0.02895, 0.02822, 0.03060, 0.02658,
     0.02759, 0.02779, 0.02575, 0.02637, 0.02603, 0.02783, 0.02525, 0.02744, 0.02429, 0.02613,
     0.02264, 0.02385, 0.02370, 0.02473, 0.02229, 0.02346, 0.02491, 0.02229, 0.02040, 0.02307,
     0.02292, 0.02118, 0.02177, 0.02030, 0.02219, 0.01899, 0.02055, 0.01909, 0.01870, 0.01938,
     0.01836, 0.01850, 0.01899, 0.01918, 0.01671, 0.01957, 0.01603, 0.01816, 0.01700, 0.01559,
     0.01773, 0.01773, 0.01685, 0.01530, 0.01612, 0.01622, 0.01467, 0.01627, 0.01564, 0.01448,
     0.01467, 0.01612, 0.01467, 0.01565, 0.01516, 0.01394, 0.01394, 0.01477, 0.01428, 0.01302,
     0.01423, 0.01146, 0.01311, 0.01418, 0.01302, 0.01224, 0.01269, 0.01190, 0.01055, 0.01258,
     0.01185, 0.01088, 0.01122, 0.01059, 0.01035, 0.01132, 0.00947, 0.01234, 0.00914, 0.01054,
     0.01054, 0.00918, 0.00841, 0.00987, 0.01030, 0.00972, 0.01035, 0.00743, 0.00860, 0.00923,
     0.00894, 0.00948, 0.00908, 0.00801, 0.00792, 0.00826, 0.00724, 0.00801, 0.00728, 0.00782,
     0.00724, 0.00777, 0.00694, 0.00811, 0.00796, 0.00665, 0.00753, 0.00724, 0.00583, 0.00656,
     0.00675, 0.00646, 0.00787, 0.00719, 0.00660, 0.00796, 0.00568, 0.00787, 0.00622, 0.00690,
     0.00694, 0.00563, 0.00597, 0.00627, 0.00660, 0.00578, 0.00719, 0.00442, 0.00524, 0.00583,
     0.00617, 0.00622, 0.00612, 0.00583, 0.00554, 0.00481, 0.00432, 0.00476, 0.00403, 0.00534,
     0.00418, 0.00490, 0.00466, 0.00534, 0.00520, 0.00379, 0.00510, 0.00422, 0.00335, 0.00427,
     0.00437, 0.00452, 0.00374, 0.00524, 0.00442, 0.00320, 0.00403, 0.00388, 0.00461, 0.00379,
     0.00345, 0.00364},

    {0.96640, 0.96400, 0.96580, 0.96400, 0.96420, 0.96330, 0.95670, 0.96270, 0.95660, 0.95880,
     0.95890, 0.95390, 0.95320, 0.95380, 0.95170, 0.94910, 0.94770, 0.94270, 0.94250, 0.94510,
     0.93970, 0.94230, 0.93810, 0.93310, 0.93570, 0.93640, 0.93310, 0.92610, 0.92570, 0.92820,
     0.92750, 0.92470, 0.91640, 0.91400, 0.91720, 0.91460, 0.91660, 0.91030, 0.90710, 0.90660,
     0.90650, 0.90360, 0.91050, 0.89860, 0.89820, 0.89550, 0.89050, 0.88670, 0.88830, 0.88260,
     0.88290, 0.87750, 0.88350, 0.87280, 0.87280, 0.86770, 0.86900, 0.86460, 0.86070, 0.86080,
     0.85960, 0.85980, 0.85630, 0.84450, 0.85140, 0.84130, 0.83630, 0.83390, 0.83410, 0.83120,
     0.82320, 0.82660, 0.83190, 0.82300, 0.81830, 0.81350, 0.80630, 0.80950, 0.81020, 0.80420,
     0.79810, 0.79170, 0.78960, 0.78720, 0.78590, 0.78610, 0.78180, 0.77540, 0.77590, 0.77300,
     0.76140, 0.76630, 0.75710, 0.75710, 0.75770, 0.74930, 0.74910, 0.74500, 0.73840, 0.73500,
     0.73370, 0.71750, 0.72810, 0.72090, 0.72030, 0.72250, 0.71240, 0.70560, 0.70260, 0.69540,
     0.70310, 0.69620, 0.68590, 0.68740, 0.67870, 0.67590, 0.66930, 0.67200, 0.66590, 0.66930,
     0.65740, 0.65780, 0.65160, 0.64690, 0.64180, 0.63860, 0.63080, 0.63760, 0.62840, 0.62330,
     0.61840, 0.63040, 0.61950, 0.61740, 0.60520, 0.60400, 0.60050, 0.59470, 0.59160, 0.58650,
     0.59390, 0.58170, 0.57710, 0.57010, 0.56440, 0.56660, 0.56190, 0.56080, 0.56090, 0.55270,
     0.55660, 0.54050, 0.53990, 0.53180, 0.53700, 0.52390, 0.52790, 0.51930, 0.51950, 0.51460,
     0.50850, 0.51290, 0.49840, 0.49800, 0.50330, 0.49200, 0.48130, 0.48150, 0.47620, 0.46580,
     0.47300, 0.47270, 0.46220, 0.45480, 0.46250, 0.44910, 0.45430, 0.45400, 0.43910, 0.43900,
     0.43930, 0.43380, 0.43720, 0.43110, 0.42150, 0.41690, 0.41710, 0.40910, 0.41680, 0.41090,
     0.40370, 0.40880, 0.39740, 0.39520, 0.38930, 0.37410, 0.38230, 0.38400, 0.37900, 0.37280,
     0.37040, 0.36430, 0.36310, 0.36480, 0.36080, 0.35330, 0.35380, 0.35410, 0.34870, 0.34340,
     0.34650, 0.33470, 0.33180, 0.32400, 0.32350, 0.32060, 0.31130, 0.32040, 0.30830, 0.31290,
     0.30530, 0.30480, 0.30300, 0.29800, 0.29560, 0.29850, 0.29100, 0.29050, 0.27730, 0.28850,
     0.28050, 0.28740, 0.27250, 0.27230, 0.26170, 0.26980, 0.26060, 0.26240, 0.26170, 0.25610,
     0.25490, 0.25530, 0.25110, 0.24920, 0.24610, 0.24430, 0.23820, 0.24380, 0.23420, 0.23300,
     0.22770, 0.22570, 0.22910, 0.22300, 0.22930, 0.21930, 0.22310, 0.22770, 0.21230, 0.21590,
     0.20360, 0.21110, 0.19670, 0.20420, 0.20590, 0.19840, 0.19580, 0.19860, 0.19390, 0.19240,
     0.18810, 0.18680, 0.19040, 0.17470, 0.17800, 0.17330, 0.17970, 0.17770, 0.17960, 0.17560,
     0.16750, 0.16600, 0.16380, 0.15960, 0.16820, 0.15580, 0.15800, 0.16110, 0.15540, 0.15520,
     0.15650, 0.15270, 0.14830, 0.15290, 0.15340, 0.13980, 0.15480, 0.13890, 0.13490, 0.13830,
     0.13390, 0.13630, 0.13300, 0.13590, 0.13870, 0.12970, 0.13070, 0.12580, 0.13560, 0.11940,
     0.12720, 0.11800, 0.12730, 0.12190, 0.11480, 0.11510, 0.11640, 0.11820, 0.10860, 0.11100,
     0.11340, 0.10590, 0.10620, 0.10410, 0.10300, 0.10270, 0.10160, 0.10430, 0.10240, 0.09500,
     0.10140, 0.10260, 0.09380, 0.10250, 0.09630, 0.09460, 0.09680, 0.09460, 0.08860, 0.09320,
     0.08820, 0.09230, 0.08740, 0.08080, 0.07720, 0.08220, 0.08350, 0.07900, 0.08160, 0.08060,
     0.07940, 0.07570, 0.07390, 0.07190, 0.07450, 0.07560, 0.07490, 0.07470, 0.06940, 0.07460,
     0.07330, 0.07110, 0.06940, 0.07250, 0.06540, 0.06410, 0.06440, 0.06660, 0.06410, 0.06520,
     0.06030, 0.06100, 0.06720, 0.06740, 0.06080, 0.05770, 0.05960, 0.05810, 0.06310, 0.05490,
     0.05680, 0.05730, 0.05310, 0.05430, 0.05360, 0.05730, 0.05200, 0.05650, 0.05010, 0.05380,
     0.04670, 0.04920, 0.04880, 0.05110, 0.04590, 0.04830, 0.05130, 0.04590, 0.04210, 0.04750,
     0.04720, 0.04370, 0.04500, 0.04180, 0.04570, 0.03910, 0.04240, 0.03930, 0.03850, 0.03990,
     0.03780, 0.03810, 0.03910, 0.03950, 0.03440, 0.04030, 0.03310, 0.03740, 0.03500, 0.03210,
     0.03650, 0.03650, 0.03470, 0.03150, 0.03320, 0.03350, 0.03020, 0.03350, 0.03220, 0.02990,
     0.03020, 0.03320, 0.03020, 0.03240, 0.03130, 0.02870, 0.02870, 0.03040, 0.02940, 0.02680,
     0.02930, 0.02360, 0.02700, 0.02920, 0.02680, 0.02520, 0.02630, 0.02450, 0.02180, 0.02590,
     0.02450, 0.02240, 0.02310, 0.02190, 0.02130, 0.02330, 0.01950, 0.02550, 0.01890, 0.02170,
     0.02170, 0.01890, 0.01740, 0.02040, 0.02120, 0.02010, 0.02140, 0.01530, 0.01770, 0.01900,
     0.01840, 0.01960, 0.01870, 0.01650, 0.01630, 0.01700, 0.01500, 0.01650, 0.01500, 0.01610,
     0.01490, 0.01600, 0.01430, 0.01670, 0.01640, 0.01370, 0.01550, 0.01490, 0.01200, 0.01350,
     0.01390, 0.01330, 0.01620, 0.01490, 0.01360, 0.01640, 0.01170, 0.01620, 0.01290, 0.01420,
     0.01430, 0.01160, 0.01230, 0.01300, 0.01360, 0.01190, 0.01490, 0.00910, 0.01080, 0.01200,
     0.01270, 0.01280, 0.01260, 0.01200, 0.01140, 0.00990, 0.00890, 0.00980, 0.00830, 0.01100,
     0.00860, 0.01010, 0.00960, 0.01100, 0.01070, 0.00780, 0.01050, 0.00870, 0.00690, 0.00880,
     0.00900, 0.00930, 0.00780, 0.01080, 0.00920, 0.00660, 0.00830, 0.00800, 0.00950, 0.00780,
     0.00710, 0.00750},

    {0.03247, 0.03470, 0.03303, 0.03470, 0.03452, 0.03535, 0.04142, 0.03591, 0.04152, 0.03950,
     0.03941, 0.04398, 0.04461, 0.04407, 0.04597, 0.04831, 0.04956, 0.05402, 0.05419, 0.05189,
     0.05666, 0.05437, 0.05807, 0.06242, 0.06017, 0.05955, 0.06242, 0.06844, 0.06878, 0.06664,
     0.06724, 0.06963, 0.07661, 0.07860, 0.07594, 0.07811, 0.07644, 0.08165, 0.08427, 0.08468,
     0.08476, 0.08711, 0.08149, 0.09112, 0.09144, 0.09358, 0.09751, 0.10046, 0.09922, 0.10362,
     0.10339, 0.10749, 0.10293, 0.11102, 0.11102, 0.11480, 0.11384, 0.11707, 0.11990, 0.11982,
     0.12069, 0.12054, 0.12305, 0.13132, 0.12652, 0.13351, 0.13690, 0.13851, 0.13838, 0.14031,
     0.14554, 0.14333, 0.13984, 0.14567, 0.14869, 0.15172, 0.15618, 0.15421, 0.15378, 0.15746,
     0.16114, 0.16491, 0.16613, 0.16752, 0.16826, 0.16815, 0.17059, 0.17416, 0.17388, 0.17547,
     0.18167, 0.17908, 0.18390, 0.18390, 0.18359, 0.18785, 0.18795, 0.18998, 0.19317, 0.19478,
     0.19538, 0.20269, 0.19797, 0.20120, 0.20147, 0.20049, 0.20489, 0.20773, 0.20895, 0.21182,
     0.20875, 0.21151, 0.21544, 0.21488, 0.21807, 0.21906, 0.22134, 0.22042, 0.22248, 0.22134,
     0.22523, 0.22510, 0.22702, 0.22842, 0.22989, 0.23079, 0.23289, 0.23107, 0.23351, 0.23480,
     0.23598, 0.23300, 0.23572, 0.23622, 0.23893, 0.23918, 0.23990, 0.24103, 0.24161, 0.24252,
     0.24118, 0.24333, 0.24406, 0.24509, 0.24585, 0.24556, 0.24617, 0.24630, 0.24629, 0.24722,
     0.24680, 0.24836, 0.24841, 0.24899, 0.24863, 0.24943, 0.24922, 0.24963, 0.24962, 0.24979,
     0.24993, 0.24983, 0.25000, 0.25000, 0.24999, 0.24994, 0.24965, 0.24966, 0.24943, 0.24883,
     0.24927, 0.24926, 0.24857, 0.24796, 0.24859, 0.24741, 0.24791, 0.24788, 0.24629, 0.24628,
     0.24632, 0.24562, 0.24606, 0.24525, 0.24384, 0.24309, 0.24313, 0.24174, 0.24308, 0.24206,
     0.24073, 0.24168, 0.23947, 0.23902, 0.23775, 0.23415, 0.23615, 0.23654, 0.23536, 0.23382,
     0.23320, 0.23159, 0.23126, 0.23172, 0.23062, 0.22848, 0.22863, 0.22871, 0.22711, 0.22548,
     0.22644, 0.22268, 0.22171, 0.21902, 0.21885, 0.21782, 0.21439, 0.21774, 0.21325, 0.21499,
     0.21209, 0.21190, 0.21119, 0.20920, 0.20822, 0.20940, 0.20632, 0.20611, 0.20040, 0.20527,
     0.20182, 0.20480, 0.19824, 0.19815, 0.19321, 0.19701, 0.19269, 0.19355, 0.19321, 0.19051,
     0.18993, 0.19012, 0.18805, 0.18710, 0.18554, 0.18462, 0.18146, 0.18436, 0.17935, 0.17871,
     0.17585, 0.17476, 0.17661, 0.17327, 0.17672, 0.17121, 0.17333, 0.17585, 0.16723, 0.16929,
     0.16215, 0.16654, 0.15801, 0.16250, 0.16351, 0.15904, 0.15746, 0.15916, 0.15630, 0.15538,
     0.15272, 0.15191, 0.15415, 0.14418, 0.14632, 0.14327, 0.14741, 0.14612, 0.14734, 0.14477,
     0.13944, 0.13844, 0.13697, 0.13413, 0.13991, 0.13153, 0.13304, 0.13515, 0.13125, 0.13111,
     0.13201, 0.12938, 0.12631, 0.12952, 0.12987, 0.12026, 0.13084, 0.11961, 0.11670, 0.11917,
     0.11597, 0.11772, 0.11531, 0.11743, 0.11946, 0.11288, 0.11362, 0.10997, 0.11721, 0.10514,
     0.11102, 0.10408, 0.11109, 0.10704, 0.10162, 0.10185, 0.10285, 0.10423, 0.09681, 0.09868,
     0.10054, 0.09469, 0.09492, 0.09326, 0.09239, 0.09215, 0.09128, 0.09342, 0.09191, 0.08597,
     0.09112, 0.09207, 0.08500, 0.09199, 0.08703, 0.08565, 0.08743, 0.08565, 0.08075, 0.08451,
     0.08042, 0.08378, 0.07976, 0.07427, 0.07124, 0.07544, 0.07653, 0.07276, 0.07494, 0.07410,
     0.07310, 0.06997, 0.06844, 0.06673, 0.06895, 0.06989, 0.06929, 0.06912, 0.06458, 0.06903,
     0.06793, 0.06605, 0.06458, 0.06724, 0.06112, 0.05999, 0.06025, 0.06216, 0.05999, 0.06095,
     0.05666, 0.05728, 0.06268, 0.06286, 0.05710, 0.05437, 0.05605, 0.05472, 0.05912, 0.05189,
     0.05357, 0.05402, 0.05028, 0.05135, 0.05073, 0.05402, 0.04930, 0.05331, 0.04759, 0.05091,
     0.04452, 0.04678, 0.04642, 0.04849, 0.04379, 0.04597, 0.04867, 0.04379, 0.04033, 0.04524,
     0.04497, 0.04179, 0.04297, 0.04005, 0.04361, 0.03757, 0.04060, 0.03776, 0.03702, 0.03831,
     0.03637, 0.03665, 0.03757, 0.03794, 0.03322, 0.03868, 0.03200, 0.03600, 0.03377, 0.03107,
     0.03517, 0.03517, 0.03350, 0.03051, 0.03210, 0.03238, 0.02929, 0.03238, 0.03116, 0.02901,
     0.02929, 0.03210, 0.02929, 0.03135, 0.03032, 0.02788, 0.02788, 0.02948, 0.02854, 0.02608,
     0.02844, 0.02304, 0.02627, 0.02835, 0.02608, 0.02457, 0.02561, 0.02390, 0.02133, 0.02523,
     0.02390, 0.02190, 0.02257, 0.02142, 0.02085, 0.02276, 0.01912, 0.02485, 0.01854, 0.02123,
     0.02123, 0.01854, 0.01710, 0.01998, 0.02075, 0.01970, 0.02094, 0.01507, 0.01739, 0.01864,
     0.01806, 0.01922, 0.01835, 0.01623, 0.01603, 0.01671, 0.01477, 0.01623, 0.01477, 0.01584,
     0.01468, 0.01574, 0.01410, 0.01642, 0.01613, 0.01351, 0.01526, 0.01468, 0.01186, 0.01332,
     0.01371, 0.01312, 0.01594, 0.01468, 0.01341, 0.01613, 0.01156, 0.01594, 0.01273, 0.01400,
     0.01410, 0.01146, 0.01215, 0.01283, 0.01341, 0.01176, 0.01468, 0.00902, 0.01068, 0.01186,
     0.01254, 0.01264, 0.01244, 0.01186, 0.01127, 0.00980, 0.00882, 0.00970, 0.00823, 0.01088,
     0.00853, 0.01000, 0.00951, 0.01088, 0.01059, 0.00774, 0.01039, 0.00862, 0.00685, 0.00872,
     0.00892, 0.00921, 0.00774, 0.01068, 0.00911, 0.00656, 0.00823, 0.00794, 0.00941, 0.00774,
     0.00705, 0.00744},

    {0.96287, 0.96035, 0.96224, 0.96035, 0.96056, 0.95961, 0.95271, 0.95899, 0.95261, 0.95490,
     0.95501, 0.94979, 0.94906, 0.94969, 0.94750, 0.94479, 0.94334, 0.93815, 0.93794, 0.94063,
     0.93503, 0.93773, 0.93338, 0.92820, 0.93089, 0.93162, 0.92820, 0.92097, 0.92056, 0.92314,
     0.92242, 0.91953, 0.91097, 0.90851, 0.91180, 0.90912, 0.91118, 0.90470, 0.90141, 0.90090,
     0.90079, 0.89782, 0.90490, 0.89268, 0.89227, 0.88950, 0.88438, 0.88049, 0.88213, 0.87629,
     0.87660, 0.87107, 0.87721, 0.86627, 0.86627, 0.86106, 0.86239, 0.85789, 0.85391, 0.85401,
     0.85279, 0.85299, 0.84942, 0.83740, 0.84443, 0.83414, 0.82905, 0.82661, 0.82681, 0.82386,
     0.81572, 0.81918, 0.82457, 0.81552, 0.81074, 0.80587, 0.79855, 0.80180, 0.80251, 0.79642,
     0.79023, 0.78374, 0.78161, 0.77918, 0.77786, 0.77806, 0.77370, 0.76722, 0.76773, 0.76479,
     0.75305, 0.75801, 0.74869, 0.74869, 0.74930, 0.74081, 0.74060, 0.73646, 0.72979, 0.72635,
     0.72504, 0.70868, 0.71938, 0.71211, 0.71150, 0.71372, 0.70353, 0.69667, 0.69364, 0.68638,
     0.69415, 0.68719, 0.67680, 0.67831, 0.66955, 0.66673, 0.66008, 0.66280, 0.65665, 0.66008,
     0.64810, 0.64850, 0.64226, 0.63753, 0.63240, 0.62918, 0.62134, 0.62818, 0.61893, 0.61380,
     0.60888, 0.62094, 0.60998, 0.60787, 0.59562, 0.59441, 0.59090, 0.58508, 0.58197, 0.57685,
     0.58427, 0.57203, 0.56742, 0.56040, 0.55468, 0.55689, 0.55217, 0.55107, 0.55117, 0.54295,
     0.54686, 0.53073, 0.53013, 0.52202, 0.52723, 0.51411, 0.51811, 0.50951, 0.50971, 0.50480,
     0.49870, 0.50310, 0.48860, 0.48820, 0.49350, 0.48220, 0.47151, 0.47171, 0.46641, 0.45602,
     0.46321, 0.46292, 0.45243, 0.44504, 0.45273, 0.43935, 0.44454, 0.44424, 0.42937, 0.42927,
     0.42957, 0.42409, 0.42748, 0.42139, 0.41182, 0.40724, 0.40744, 0.39946, 0.40714, 0.40126,
     0.39408, 0.39916, 0.38781, 0.38562, 0.37974, 0.36462, 0.37278, 0.37447, 0.36949, 0.36332,
     0.36094, 0.35487, 0.35367, 0.35536, 0.35139, 0.34393, 0.34443, 0.34473, 0.33936, 0.33409,
     0.33717, 0.32545, 0.32257, 0.31483, 0.31433, 0.31145, 0.30223, 0.31125, 0.29925, 0.30381,
     0.29627, 0.29578, 0.29399, 0.28903, 0.28666, 0.28953, 0.28210, 0.28160, 0.26853, 0.27962,
     0.27170, 0.27853, 0.26377, 0.26358, 0.25309, 0.26110, 0.25200, 0.25378, 0.25309, 0.24754,
     0.24636, 0.24675, 0.24260, 0.24072, 0.23766, 0.23588, 0.22985, 0.23538, 0.22590, 0.22471,
     0.21948, 0.21751, 0.22086, 0.21484, 0.22106, 0.21119, 0.21494, 0.21948, 0.20428, 0.20784,
     0.19571, 0.20310, 0.18891, 0.19630, 0.19798, 0.19058, 0.18802, 0.19078, 0.18615, 0.18467,
     0.18044, 0.17916, 0.18271, 0.16726, 0.17050, 0.16588, 0.17217, 0.17021, 0.17208, 0.16814,
     0.16018, 0.15871, 0.15655, 0.15242, 0.16087, 0.14869, 0.15085, 0.15390, 0.14830, 0.14810,
     0.14938, 0.14565, 0.14133, 0.14585, 0.14634, 0.13300, 0.14771, 0.13212, 0.12820, 0.13153,
     0.12723, 0.12957, 0.12634, 0.12918, 0.13193, 0.12312, 0.12409, 0.11930, 0.12889, 0.11305,
     0.12067, 0.11168, 0.12077, 0.11549, 0.10855, 0.10884, 0.11011, 0.11187, 0.10250, 0.10484,
     0.10719, 0.09987, 0.10016, 0.09811, 0.09704, 0.09675, 0.09568, 0.09831, 0.09646, 0.08925,
     0.09548, 0.09665, 0.08809, 0.09656, 0.09052, 0.08886, 0.09101, 0.08886, 0.08303, 0.08750,
     0.08264, 0.08663, 0.08186, 0.07546, 0.07197, 0.07682, 0.07808, 0.07371, 0.07623, 0.07526,
     0.07410, 0.07051, 0.06877, 0.06684, 0.06935, 0.07042, 0.06974, 0.06955, 0.06442, 0.06945,
     0.06819, 0.06606, 0.06442, 0.06742, 0.06055, 0.05930, 0.05959, 0.06171, 0.05930, 0.06036,
     0.05563, 0.05631, 0.06229, 0.06249, 0.05612, 0.05313, 0.05496, 0.05352, 0.05833, 0.05044,
     0.05226, 0.05275, 0.04870, 0.04986, 0.04919, 0.05275, 0.04765, 0.05198, 0.04582, 0.04938,
     0.04256, 0.04496, 0.04458, 0.04678, 0.04180, 0.04410, 0.04698, 0.04180, 0.03816, 0.04333,
     0.04304, 0.03969, 0.04094, 0.03788, 0.04161, 0.03530, 0.03845, 0.03549, 0.03473, 0.03606,
     0.03406, 0.03435, 0.03530, 0.03568, 0.03083, 0.03644, 0.02959, 0.03368, 0.03140, 0.02865,
     0.03282, 0.03282, 0.03111, 0.02808, 0.02969, 0.02997, 0.02685, 0.02997, 0.02874, 0.02656,
     0.02685, 0.02969, 0.02685, 0.02893, 0.02789, 0.02543, 0.02543, 0.02704, 0.02609, 0.02363,
     0.02600, 0.02063, 0.02382, 0.02590, 0.02363, 0.02213, 0.02316, 0.02147, 0.01894, 0.02279,
     0.02147, 0.01950, 0.02016, 0.01903, 0.01847, 0.02034, 0.01679, 0.02241, 0.01623, 0.01884,
     0.01884, 0.01623, 0.01484, 0.01763, 0.01838, 0.01735, 0.01856, 0.01289, 0.01512, 0.01632,
     0.01577, 0.01688, 0.01605, 0.01400, 0.01382, 0.01447, 0.01262, 0.01400, 0.01262, 0.01363,
     0.01252, 0.01354, 0.01197, 0.01419, 0.01391, 0.01142, 0.01308, 0.01252, 0.00987, 0.01124,
     0.01161, 0.01106, 0.01373, 0.01252, 0.01133, 0.01391, 0.00959, 0.01373, 0.01069, 0.01188,
     0.01197, 0.00950, 0.01014, 0.01078, 0.01133, 0.00978, 0.01252, 0.00724, 0.00877, 0.00987,
     0.01051, 0.01060, 0.01041, 0.00987, 0.00932, 0.00796, 0.00706, 0.00787, 0.00652, 0.00896,
     0.00679, 0.00814, 0.00769, 0.00896, 0.00868, 0.00608, 0.00850, 0.00688, 0.00528, 0.00697,
     0.00715, 0.00742, 0.00608, 0.00877, 0.00733, 0.00501, 0.00652, 0.00625, 0.00760, 0.00608,
     0.00545, 0.00581},

    {0.96993, 0.96765, 0.96936, 0.96765, 0.96784, 0.96698, 0.96069, 0.96641, 0.96059, 0.96270,
     0.96279, 0.95801, 0.95734, 0.95791, 0.95590, 0.95341, 0.95206, 0.94725, 0.94706, 0.94956,
     0.94437, 0.94687, 0.94282, 0.93800, 0.94051, 0.94118, 0.93800, 0.93123, 0.93084, 0.93326,
     0.93258, 0.92987, 0.92183, 0.91949, 0.92260, 0.92008, 0.92202, 0.91590, 0.91279, 0.91230,
     0.91221, 0.90938, 0.91609, 0.90452, 0.90413, 0.90150, 0.89662, 0.89291, 0.89447, 0.88891,
     0.88920, 0.88393, 0.88979, 0.87933, 0.87933, 0.87434, 0.87561, 0.87131, 0.86749, 0.86759,
     0.86641, 0.86660, 0.86318, 0.85160, 0.85837, 0.84846, 0.84355, 0.84120, 0.84139, 0.83854,
     0.83068, 0.83402, 0.83923, 0.83048, 0.82586, 0.82113, 0.81405, 0.81720, 0.81789, 0.81198,
     0.80597, 0.79966, 0.79759, 0.79522, 0.79394, 0.79414, 0.78990, 0.78358, 0.78407, 0.78121,
     0.76975, 0.77459, 0.76550, 0.76550, 0.76610, 0.75779, 0.75760, 0.75354, 0.74701, 0.74365,
     0.74236, 0.72632, 0.73682, 0.72969, 0.72910, 0.73128, 0.72127, 0.71453, 0.71156, 0.70442,
     0.71205, 0.70521, 0.69500, 0.69649, 0.68785, 0.68507, 0.67852, 0.68120, 0.67514, 0.67852,
     0.66670, 0.66710, 0.66094, 0.65627, 0.65120, 0.64802, 0.64026, 0.64702, 0.63787, 0.63280,
     0.62792, 0.63986, 0.62902, 0.62693, 0.61478, 0.61359, 0.61010, 0.60432, 0.60123, 0.59615,
     0.60353, 0.59137, 0.58678, 0.57980, 0.57412, 0.57631, 0.57163, 0.57053, 0.57063, 0.56244,
     0.56634, 0.55027, 0.54967, 0.54158, 0.54677, 0.53369, 0.53768, 0.52909, 0.52929, 0.52440,
     0.51830, 0.52270, 0.50820, 0.50780, 0.51310, 0.50180, 0.49109, 0.49129, 0.48599, 0.47558,
     0.48279, 0.48248, 0.47197, 0.46456, 0.47227, 0.45885, 0.46406, 0.46376, 0.44883, 0.44873,
     0.44903, 0.44351, 0.44692, 0.44081, 0.43118, 0.42656, 0.42676, 0.41874, 0.42646, 0.42054,
     0.41332, 0.41844, 0.40699, 0.40478, 0.39886, 0.38358, 0.39182, 0.39353, 0.38851, 0.38228,
     0.37987, 0.37373, 0.37253, 0.37423, 0.37021, 0.36267, 0.36317, 0.36347, 0.35804, 0.35271,
     0.35583, 0.34395, 0.34103, 0.33317, 0.33267, 0.32975, 0.32038, 0.32955, 0.31735, 0.32199,
     0.31433, 0.31382, 0.31201, 0.30696, 0.30454, 0.30747, 0.29990, 0.29940, 0.28607, 0.29738,
     0.28930, 0.29627, 0.28123, 0.28103, 0.27032, 0.27850, 0.26920, 0.27102, 0.27032, 0.26465,
     0.26344, 0.26385, 0.25960, 0.25768, 0.25454, 0.25272, 0.24655, 0.25222, 0.24250, 0.24129,
     0.23592, 0.23389, 0.23734, 0.23116, 0.23754, 0.22741, 0.23126, 0.23592, 0.22032, 0.22396,
     0.21149, 0.21910, 0.20449, 0.21210, 0.21382, 0.20622, 0.20358, 0.20642, 0.20165, 0.20013,
     0.19576, 0.19444, 0.19809, 0.18214, 0.18550, 0.18072, 0.18723, 0.18519, 0.18712, 0.18306,
     0.17482, 0.17329, 0.17105, 0.16678, 0.17553, 0.16291, 0.16515, 0.16831, 0.16250, 0.16230,
     0.16362, 0.15975, 0.15527, 0.15995, 0.16046, 0.14660, 0.16189, 0.14568, 0.14160, 0.14507,
     0.14058, 0.14303, 0.13966, 0.14262, 0.14547, 0.13628, 0.13731, 0.13230, 0.14231, 0.12576,
     0.13373, 0.12432, 0.13383, 0.12831, 0.12105, 0.12136, 0.12269, 0.12453, 0.11470, 0.11716,
     0.11961, 0.11193, 0.11224, 0.11009, 0.10896, 0.10865, 0.10752, 0.11029, 0.10834, 0.10075,
     0.10732, 0.10855, 0.09951, 0.10844, 0.10208, 0.10034, 0.10260, 0.10034, 0.09417, 0.09890,
     0.09376, 0.09797, 0.09294, 0.08614, 0.08243, 0.08758, 0.08892, 0.08429, 0.08697, 0.08594,
     0.08470, 0.08088, 0.07903, 0.07696, 0.07965, 0.08078, 0.08006, 0.07985, 0.07438, 0.07975,
     0.07841, 0.07614, 0.07438, 0.07758, 0.07025, 0.06890, 0.06921, 0.07149, 0.06890, 0.07004,
     0.06497, 0.06569, 0.07211, 0.07231, 0.06548, 0.06227, 0.06424, 0.06269, 0.06787, 0.05937,
     0.06134, 0.06186, 0.05749, 0.05874, 0.05801, 0.06186, 0.05635, 0.06103, 0.05438, 0.05822,
     0.05084, 0.05344, 0.05302, 0.05542, 0.05000, 0.05250, 0.05562, 0.05000, 0.04604, 0.05167,
     0.05136, 0.04771, 0.04906, 0.04572, 0.04979, 0.04290, 0.04635, 0.04311, 0.04227, 0.04374,
     0.04154, 0.04185, 0.04290, 0.04332, 0.03797, 0.04415, 0.03661, 0.04112, 0.03860, 0.03556,
     0.04018, 0.04018, 0.03829, 0.03492, 0.03671, 0.03703, 0.03355, 0.03703, 0.03566, 0.03324,
     0.03355, 0.03671, 0.03355, 0.03587, 0.03471, 0.03197, 0.03197, 0.03377, 0.03271, 0.02996,
     0.03261, 0.02658, 0.03018, 0.03250, 0.02996, 0.02827, 0.02944, 0.02753, 0.02466, 0.02901,
     0.02753, 0.02530, 0.02604, 0.02477, 0.02413, 0.02626, 0.02221, 0.02859, 0.02157, 0.02456,
     0.02456, 0.02157, 0.01996, 0.02317, 0.02402, 0.02285, 0.02424, 0.01771, 0.02028, 0.02168,
     0.02103, 0.02232, 0.02135, 0.01900, 0.01878, 0.01953, 0.01738, 0.01900, 0.01738, 0.01857,
     0.01727, 0.01846, 0.01663, 0.01921, 0.01889, 0.01598, 0.01792, 0.01727, 0.01413, 0.01576,
     0.01620, 0.01554, 0.01867, 0.01727, 0.01587, 0.01889, 0.01381, 0.01867, 0.01511, 0.01652,
     0.01663, 0.01370, 0.01446, 0.01522, 0.01587, 0.01402, 0.01727, 0.01096, 0.01283, 0.01413,
     0.01490, 0.01500, 0.01479, 0.01413, 0.01348, 0.01184, 0.01074, 0.01173, 0.01008, 0.01304,
     0.01041, 0.01206, 0.01151, 0.01304, 0.01272, 0.00952, 0.01250, 0.01052, 0.00852, 0.01063,
     0.01085, 0.01118, 0.00952, 0.01283, 0.01107, 0.00819, 0.01008, 0.00975, 0.01140, 0.00952,
     0.00875, 0.00919},
};

static const double modulation6[6][626] = {

    {65.00,  65.10,  65.20,  65.30,  65.40,  65.50,  65.60,  65.70,  65.80,  65.90,  66.00,  66.10,
     66.20,  66.30,  66.40,  66.50,  66.60,  66.70,  66.80,  66.90,  67.00,  67.10,  67.20,  67.30,
     67.40,  67.50,  67.60,  67.70,  67.80,  67.90,  68.00,  68.10,  68.20,  68.30,  68.40,  68.50,
     68.60,  68.70,  68.80,  68.90,  69.00,  69.10,  69.20,  69.30,  69.40,  69.50,  69.60,  69.70,
     69.80,  69.90,  70.00,  70.10,  70.20,  70.30,  70.40,  70.50,  70.60,  70.70,  70.80,  70.90,
     71.00,  71.10,  71.20,  71.30,  71.40,  71.50,  71.60,  71.70,  71.80,  71.90,  72.00,  72.10,
     72.20,  72.30,  72.40,  72.50,  72.60,  72.70,  72.80,  72.90,  73.00,  73.10,  73.20,  73.30,
     73.40,  73.50,  73.60,  73.70,  73.80,  73.90,  74.00,  74.10,  74.20,  74.30,  74.40,  74.50,
     74.60,  74.70,  74.80,  74.90,  75.00,  75.10,  75.20,  75.30,  75.40,  75.50,  75.60,  75.70,
     75.80,  75.90,  76.00,  76.10,  76.20,  76.30,  76.40,  76.50,  76.60,  76.70,  76.80,  76.90,
     77.00,  77.10,  77.20,  77.30,  77.40,  77.50,  77.60,  77.70,  77.80,  77.90,  78.00,  78.10,
     78.20,  78.30,  78.40,  78.50,  78.60,  78.70,  78.80,  78.90,  79.00,  79.10,  79.20,  79.30,
     79.40,  79.50,  79.60,  79.70,  79.80,  79.90,  80.00,  80.10,  80.20,  80.30,  80.40,  80.50,
     80.60,  80.70,  80.80,  80.90,  81.00,  81.10,  81.20,  81.30,  81.40,  81.50,  81.60,  81.70,
     81.80,  81.90,  82.00,  82.10,  82.20,  82.30,  82.40,  82.50,  82.60,  82.70,  82.80,  82.90,
     83.00,  83.10,  83.20,  83.30,  83.40,  83.50,  83.60,  83.70,  83.80,  83.90,  84.00,  84.10,
     84.20,  84.30,  84.40,  84.50,  84.60,  84.70,  84.80,  84.90,  85.00,  85.10,  85.20,  85.30,
     85.40,  85.50,  85.60,  85.70,  85.80,  85.90,  86.00,  86.10,  86.20,  86.30,  86.40,  86.50,
     86.60,  86.70,  86.80,  86.90,  87.00,  87.10,  87.20,  87.30,  87.40,  87.50,  87.60,  87.70,
     87.80,  87.90,  88.00,  88.10,  88.20,  88.30,  88.40,  88.50,  88.60,  88.70,  88.80,  88.90,
     89.00,  89.10,  89.20,  89.30,  89.40,  89.50,  89.60,  89.70,  89.80,  89.90,  90.00,  90.10,
     90.20,  90.30,  90.40,  90.50,  90.60,  90.70,  90.80,  90.90,  91.00,  91.10,  91.20,  91.30,
     91.40,  91.50,  91.60,  91.70,  91.80,  91.90,  92.00,  92.10,  92.20,  92.30,  92.40,  92.50,
     92.60,  92.70,  92.80,  92.90,  93.00,  93.10,  93.20,  93.30,  93.40,  93.50,  93.60,  93.70,
     93.80,  93.90,  94.00,  94.10,  94.20,  94.30,  94.40,  94.50,  94.60,  94.70,  94.80,  94.90,
     95.00,  95.10,  95.20,  95.30,  95.40,  95.50,  95.60,  95.70,  95.80,  95.90,  96.00,  96.10,
     96.20,  96.30,  96.40,  96.50,  96.60,  96.70,  96.80,  96.90,  97.00,  97.10,  97.20,  97.30,
     97.40,  97.50,  97.60,  97.70,  97.80,  97.90,  98.00,  98.10,  98.20,  98.30,  98.40,  98.50,
     98.60,  98.70,  98.80,  98.90,  99.00,  99.10,  99.20,  99.30,  99.40,  99.50,  99.60,  99.70,
     99.80,  99.90,  100.00, 100.10, 100.20, 100.30, 100.40, 100.50, 100.60, 100.70, 100.80, 100.90,
     101.00, 101.10, 101.20, 101.30, 101.40, 101.50, 101.60, 101.70, 101.80, 101.90, 102.00, 102.10,
     102.20, 102.30, 102.40, 102.50, 102.60, 102.70, 102.80, 102.90, 103.00, 103.10, 103.20, 103.30,
     103.40, 103.50, 103.60, 103.70, 103.80, 103.90, 104.00, 104.10, 104.20, 104.30, 104.40, 104.50,
     104.60, 104.70, 104.80, 104.90, 105.00, 105.10, 105.20, 105.30, 105.40, 105.50, 105.60, 105.70,
     105.80, 105.90, 106.00, 106.10, 106.20, 106.30, 106.40, 106.50, 106.60, 106.70, 106.80, 106.90,
     107.00, 107.10, 107.20, 107.30, 107.40, 107.50, 107.60, 107.70, 107.80, 107.90, 108.00, 108.10,
     108.20, 108.30, 108.40, 108.50, 108.60, 108.70, 108.80, 108.90, 109.00, 109.10, 109.20, 109.30,
     109.40, 109.50, 109.60, 109.70, 109.80, 109.90, 110.00, 110.10, 110.20, 110.30, 110.40, 110.50,
     110.60, 110.70, 110.80, 110.90, 111.00, 111.10, 111.20, 111.30, 111.40, 111.50, 111.60, 111.70,
     111.80, 111.90, 112.00, 112.10, 112.20, 112.30, 112.40, 112.50, 112.60, 112.70, 112.80, 112.90,
     113.00, 113.10, 113.20, 113.30, 113.40, 113.50, 113.60, 113.70, 113.80, 113.90, 114.00, 114.10,
     114.20, 114.30, 114.40, 114.50, 114.60, 114.70, 114.80, 114.90, 115.00, 115.10, 115.20, 115.30,
     115.40, 115.50, 115.60, 115.70, 115.80, 115.90, 116.00, 116.10, 116.20, 116.30, 116.40, 116.50,
     116.60, 116.70, 116.80, 116.90, 117.00, 117.10, 117.20, 117.30, 117.40, 117.50, 117.60, 117.70,
     117.80, 117.90, 118.00, 118.10, 118.20, 118.30, 118.40, 118.50, 118.60, 118.70, 118.80, 118.90,
     119.00, 119.10, 119.20, 119.30, 119.40, 119.50, 119.60, 119.70, 119.80, 119.90, 120.00, 120.10,
     120.20, 120.30, 120.40, 120.50, 120.60, 120.70, 120.80, 120.90, 121.00, 121.10, 121.20, 121.30,
     121.40, 121.50, 121.60, 121.70, 121.80, 121.90, 122.00, 122.10, 122.20, 122.30, 122.40, 122.50,
     122.60, 122.70, 122.80, 122.90, 123.00, 124.00, 124.10, 124.20, 124.30, 124.40, 124.50, 124.60,
     124.70, 124.80, 124.90, 125.00, 125.10, 125.20, 125.30, 125.40, 125.50, 125.60, 125.70},

    {0.46118, 0.46267, 0.46260, 0.46062, 0.45988, 0.45750, 0.46021, 0.45820, 0.45774, 0.45697,
     0.45678, 0.45633, 0.45606, 0.45543, 0.45327, 0.45136, 0.45307, 0.45066, 0.45022, 0.45080,
     0.45273, 0.44948, 0.44888, 0.44860, 0.44675, 0.44709, 0.44582, 0.44565, 0.44609, 0.44360,
     0.44545, 0.44266, 0.44093, 0.43993, 0.44263, 0.43984, 0.44125, 0.43735, 0.43871, 0.43871,
     0.43657, 0.43548, 0.43303, 0.43389, 0.43231, 0.43082, 0.42820, 0.43007, 0.42968, 0.42675,
     0.42627, 0.42764, 0.42337, 0.42284, 0.42278, 0.42405, 0.42300, 0.42091, 0.42366, 0.41906,
     0.41808, 0.41793, 0.41369, 0.41378, 0.41458, 0.40974, 0.41490, 0.40885, 0.40906, 0.40843,
     0.40971, 0.40465, 0.40670, 0.40583, 0.40500, 0.40214, 0.39765, 0.39891, 0.39995, 0.39777,
     0.39554, 0.39643, 0.39355, 0.39314, 0.39298, 0.39014, 0.39016, 0.38465, 0.38676, 0.38720,
     0.38353, 0.38203, 0.38124, 0.37889, 0.37913, 0.38090, 0.37601, 0.37684, 0.37213, 0.37500,
     0.36958, 0.37097, 0.36830, 0.36682, 0.36694, 0.36576, 0.36606, 0.36664, 0.36577, 0.35699,
     0.35730, 0.36052, 0.35415, 0.35786, 0.35534, 0.35470, 0.35333, 0.34656, 0.34505, 0.34741,
     0.34548, 0.34278, 0.34095, 0.34258, 0.34055, 0.33822, 0.34134, 0.33565, 0.33245, 0.33118,
     0.33255, 0.33021, 0.33177, 0.32911, 0.32361, 0.32331, 0.32341, 0.31790, 0.32211, 0.31771,
     0.31716, 0.31626, 0.31212, 0.31109, 0.31258, 0.31131, 0.30939, 0.30945, 0.30670, 0.30392,
     0.30476, 0.29625, 0.29820, 0.29838, 0.29118, 0.29760, 0.29542, 0.29289, 0.29306, 0.29124,
     0.28701, 0.29202, 0.28291, 0.28125, 0.28487, 0.28201, 0.28330, 0.27724, 0.27599, 0.27545,
     0.27225, 0.26982, 0.27139, 0.27038, 0.26739, 0.27304, 0.26271, 0.26461, 0.26286, 0.26207,
     0.25751, 0.25496, 0.25117, 0.25785, 0.25969, 0.25750, 0.25104, 0.24651, 0.24727, 0.24331,
     0.24747, 0.24436, 0.24197, 0.24228, 0.23812, 0.23746, 0.24105, 0.23958, 0.23261, 0.23509,
     0.23063, 0.22904, 0.22771, 0.22589, 0.22494, 0.22919, 0.21779, 0.22510, 0.22190, 0.22851,
     0.22021, 0.21828, 0.21555, 0.21002, 0.21568, 0.21790, 0.21169, 0.20729, 0.20910, 0.20595,
     0.20535, 0.20659, 0.20241, 0.19891, 0.20217, 0.19881, 0.19875, 0.19408, 0.19711, 0.19132,
     0.19490, 0.19370, 0.19445, 0.18866, 0.19095, 0.18546, 0.18449, 0.18135, 0.18628, 0.18052,
     0.18081, 0.18175, 0.17860, 0.18217, 0.17379, 0.17589, 0.17549, 0.17115, 0.17109, 0.16823,
     0.16760, 0.16847, 0.16650, 0.16673, 0.16417, 0.15719, 0.16389, 0.16475, 0.16396, 0.15941,
     0.15694, 0.15821, 0.15311, 0.15547, 0.15958, 0.15608, 0.15193, 0.15027, 0.15130, 0.15188,
     0.14897, 0.14370, 0.14713, 0.14235, 0.13992, 0.13778, 0.14128, 0.14296, 0.13726, 0.13844,
     0.13587, 0.13568, 0.13714, 0.13412, 0.13302, 0.13634, 0.13058, 0.13133, 0.13256, 0.12865,
     0.12876, 0.12829, 0.12406, 0.12533, 0.12706, 0.12624, 0.11936, 0.11965, 0.11939, 0.11757,
     0.11776, 0.11811, 0.11718, 0.11292, 0.11913, 0.10995, 0.11550, 0.11022, 0.11052, 0.10881,
     0.10867, 0.10698, 0.10993, 0.10747, 0.10380, 0.10409, 0.10705, 0.10460, 0.10166, 0.10276,
     0.10485, 0.09890, 0.10002, 0.09657, 0.09693, 0.09508, 0.09633, 0.09700, 0.09512, 0.09168,
     0.09455, 0.09765, 0.09106, 0.09576, 0.09068, 0.09121, 0.09251, 0.08922, 0.08709, 0.08406,
     0.08608, 0.08811, 0.08714, 0.08411, 0.08723, 0.08366, 0.08314, 0.08139, 0.08106, 0.08273,
     0.07820, 0.08163, 0.07717, 0.07587, 0.07916, 0.07616, 0.07180, 0.07727, 0.07398, 0.07423,
     0.07447, 0.07200, 0.07306, 0.06731, 0.07200, 0.06953, 0.07260, 0.07068, 0.06838, 0.06614,
     0.06788, 0.06488, 0.06502, 0.06604, 0.06503, 0.06189, 0.06450, 0.06353, 0.06871, 0.06509,
     0.06091, 0.06271, 0.06141, 0.05912, 0.06207, 0.05918, 0.05966, 0.05936, 0.05913, 0.05385,
     0.05791, 0.05680, 0.05699, 0.05690, 0.05578, 0.05569, 0.05763, 0.05589, 0.05162, 0.05217,
     0.05249, 0.05286, 0.05150, 0.05125, 0.05162, 0.04853, 0.05216, 0.04819, 0.05178, 0.04746,
     0.04974, 0.05071, 0.04636, 0.04751, 0.04621, 0.04922, 0.04485, 0.04282, 0.04456, 0.04577,
     0.04733, 0.04461, 0.04297, 0.03982, 0.04220, 0.04204, 0.04470, 0.03948, 0.04224, 0.03938,
     0.04113, 0.03856, 0.03958, 0.03687, 0.03819, 0.03968, 0.03604, 0.03745, 0.03770, 0.03881,
     0.03740, 0.03711, 0.03711, 0.03750, 0.03827, 0.03633, 0.03664, 0.03596, 0.03561, 0.03440,
     0.03416, 0.03353, 0.03344, 0.03290, 0.03280, 0.03212, 0.03479, 0.03208, 0.03169, 0.03309,
     0.03087, 0.03068, 0.03111, 0.03014, 0.03140, 0.03072, 0.03159, 0.02948, 0.02990, 0.03048,
     0.03073, 0.02976, 0.02836, 0.03049, 0.02816, 0.02985, 0.02744, 0.02492, 0.02753, 0.02632,
     0.02492, 0.02622, 0.02767, 0.02521, 0.02632, 0.02540, 0.02395, 0.02473, 0.02274, 0.02603,
     0.02361, 0.02216, 0.02414, 0.02342, 0.02327, 0.02178, 0.02352, 0.02192, 0.02400, 0.02081,
     0.02298, 0.01936, 0.02235, 0.02110, 0.02090, 0.02042, 0.02318, 0.02076, 0.01984, 0.01979,
     0.02071, 0.01921, 0.02177, 0.02046, 0.01824, 0.01985, 0.01935, 0.01999, 0.01950, 0.02037,
     0.01824, 0.01713, 0.01912, 0.01771, 0.01959, 0.01809, 0.01877, 0.01690, 0.01863, 0.01684,
     0.01844, 0.01568, 0.01858, 0.01810, 0.01786, 0.01757, 0.01451, 0.01602, 0.01583, 0.01534,
     0.01389, 0.01630, 0.01461, 0.01572, 0.01592, 0.01519, 0.01655, 0.01476, 0.01345, 0.01369,
     0.01722, 0.01485, 0.01427, 0.01345, 0.01456, 0.01480, 0.01374, 0.01747, 0.01403, 0.01456,
     0.01442, 0.01209, 0.01239, 0.01311, 0.01268, 0.01253, 0.01195, 0.01306, 0.01249, 0.01355,
     0.01229, 0.01413, 0.01359, 0.01127, 0.01166, 0.01045, 0.01079, 0.01041, 0.01350, 0.01326,
     0.01060, 0.00977, 0.00905, 0.00856, 0.01248, 0.00943, 0.00992, 0.00987, 0.01074, 0.00943,
     0.01185, 0.01118, 0.00890, 0.00986, 0.00929, 0.00822, 0.00939, 0.00827, 0.00818},

    {0.95430, 0.95690, 0.95710, 0.95300, 0.95170, 0.94690, 0.95250, 0.94820, 0.94680, 0.94550,
     0.94480, 0.94400, 0.94340, 0.94230, 0.93770, 0.93400, 0.93730, 0.93230, 0.93170, 0.93250,
     0.93700, 0.93000, 0.92880, 0.92830, 0.92400, 0.92500, 0.92250, 0.92190, 0.92260, 0.91810,
     0.92150, 0.91580, 0.91230, 0.91040, 0.91540, 0.91020, 0.91280, 0.90520, 0.90780, 0.90810,
     0.90300, 0.90080, 0.89620, 0.89770, 0.89490, 0.89140, 0.88610, 0.88970, 0.88870, 0.88260,
     0.88200, 0.88520, 0.87580, 0.87510, 0.87460, 0.87710, 0.87520, 0.87080, 0.87690, 0.86720,
     0.86490, 0.86480, 0.85620, 0.85650, 0.85760, 0.84760, 0.85830, 0.84620, 0.84630, 0.84530,
     0.84780, 0.83740, 0.84160, 0.83980, 0.83790, 0.83190, 0.82300, 0.82610, 0.82790, 0.82320,
     0.81820, 0.82010, 0.81460, 0.81430, 0.81360, 0.80740, 0.80730, 0.79670, 0.80050, 0.80120,
     0.79360, 0.79050, 0.78920, 0.78410, 0.78480, 0.78850, 0.77820, 0.78020, 0.77010, 0.77600,
     0.76480, 0.76720, 0.76240, 0.75910, 0.75940, 0.75700, 0.75770, 0.75840, 0.75710, 0.73930,
     0.73940, 0.74600, 0.73270, 0.74030, 0.73500, 0.73360, 0.73120, 0.71730, 0.71400, 0.71880,
     0.71490, 0.70910, 0.70550, 0.70900, 0.70450, 0.70000, 0.70600, 0.69440, 0.68800, 0.68560,
     0.68800, 0.68340, 0.68620, 0.68080, 0.66950, 0.66890, 0.66940, 0.65770, 0.66640, 0.65780,
     0.65660, 0.65440, 0.64580, 0.64390, 0.64670, 0.64380, 0.64030, 0.64000, 0.63430, 0.62900,
     0.63060, 0.61280, 0.61750, 0.61770, 0.60290, 0.61590, 0.61130, 0.60600, 0.60600, 0.60260,
     0.59420, 0.60440, 0.58570, 0.58170, 0.58960, 0.58370, 0.58660, 0.57360, 0.57090, 0.56980,
     0.56350, 0.55850, 0.56160, 0.55940, 0.55320, 0.56540, 0.54380, 0.54730, 0.54410, 0.54220,
     0.53280, 0.52760, 0.51980, 0.53340, 0.53730, 0.53260, 0.51970, 0.50970, 0.51180, 0.50390,
     0.51240, 0.50590, 0.50140, 0.50140, 0.49290, 0.49120, 0.49880, 0.49580, 0.48140, 0.48620,
     0.47760, 0.47370, 0.47130, 0.46800, 0.46530, 0.47400, 0.45070, 0.46620, 0.45950, 0.47260,
     0.45560, 0.45180, 0.44590, 0.43500, 0.44630, 0.45130, 0.43800, 0.42900, 0.43270, 0.42640,
     0.42490, 0.42720, 0.41910, 0.41150, 0.41870, 0.41130, 0.41100, 0.40170, 0.40770, 0.39580,
     0.40330, 0.40110, 0.40260, 0.39040, 0.39520, 0.38380, 0.38170, 0.37530, 0.38540, 0.37330,
     0.37410, 0.37630, 0.36970, 0.37700, 0.35940, 0.36390, 0.36300, 0.35430, 0.35430, 0.34790,
     0.34680, 0.34850, 0.34450, 0.34470, 0.33970, 0.32510, 0.33920, 0.34090, 0.33960, 0.33040,
     0.32500, 0.32710, 0.31720, 0.32180, 0.33020, 0.32280, 0.31430, 0.31060, 0.31280, 0.31410,
     0.30810, 0.29720, 0.30420, 0.29460, 0.28930, 0.28550, 0.29230, 0.29610, 0.28380, 0.28660,
     0.28110, 0.28090, 0.28390, 0.27740, 0.27540, 0.28190, 0.27000, 0.27190, 0.27410, 0.26600,
     0.26640, 0.26560, 0.25670, 0.25940, 0.26290, 0.26130, 0.24690, 0.24740, 0.24730, 0.24320,
     0.24350, 0.24440, 0.24220, 0.23360, 0.24650, 0.22780, 0.23900, 0.22790, 0.22890, 0.22520,
     0.22490, 0.22140, 0.22740, 0.22250, 0.21510, 0.21580, 0.22200, 0.21630, 0.21040, 0.21240,
     0.21700, 0.20470, 0.20710, 0.19970, 0.20080, 0.19680, 0.19940, 0.20060, 0.19670, 0.18960,
     0.19590, 0.20210, 0.18840, 0.19820, 0.18790, 0.18880, 0.19140, 0.18470, 0.18020, 0.17380,
     0.17820, 0.18240, 0.18040, 0.17400, 0.18040, 0.17310, 0.17220, 0.16860, 0.16790, 0.17110,
     0.16200, 0.16890, 0.15960, 0.15690, 0.16380, 0.15770, 0.14860, 0.15980, 0.15320, 0.15370,
     0.15430, 0.14900, 0.15110, 0.13940, 0.14900, 0.14390, 0.15050, 0.14610, 0.14160, 0.13680,
     0.14040, 0.13420, 0.13440, 0.13650, 0.13460, 0.12820, 0.13350, 0.13150, 0.14210, 0.13480,
     0.12600, 0.12990, 0.12710, 0.12230, 0.12840, 0.12250, 0.12340, 0.12270, 0.12240, 0.11140,
     0.11970, 0.11740, 0.11780, 0.11780, 0.11530, 0.11520, 0.11940, 0.11570, 0.10680, 0.10820,
     0.10850, 0.10980, 0.10690, 0.10620, 0.10680, 0.10040, 0.10790, 0.09990, 0.10720, 0.09820,
     0.10300, 0.10490, 0.09600, 0.09820, 0.09570, 0.10200, 0.09280, 0.08860, 0.09210, 0.09460,
     0.09810, 0.09240, 0.08890, 0.08230, 0.08740, 0.08690, 0.09240, 0.08180, 0.08750, 0.08140,
     0.08510, 0.07970, 0.08190, 0.07620, 0.07920, 0.08220, 0.07450, 0.07750, 0.07810, 0.08040,
     0.07740, 0.07670, 0.07670, 0.07760, 0.07920, 0.07510, 0.07610, 0.07460, 0.07360, 0.07120,
     0.07060, 0.06940, 0.06920, 0.06810, 0.06780, 0.06640, 0.07210, 0.06630, 0.06550, 0.06840,
     0.06390, 0.06350, 0.06430, 0.06230, 0.06490, 0.06350, 0.06530, 0.06120, 0.06180, 0.06300,
     0.06360, 0.06160, 0.05880, 0.06320, 0.05830, 0.06170, 0.05680, 0.05150, 0.05690, 0.05450,
     0.05150, 0.05430, 0.05720, 0.05210, 0.05450, 0.05260, 0.04950, 0.05120, 0.04700, 0.05380,
     0.04880, 0.04590, 0.05000, 0.04840, 0.04810, 0.04520, 0.04870, 0.04530, 0.04960, 0.04310,
     0.04750, 0.04020, 0.04630, 0.04370, 0.04320, 0.04230, 0.04800, 0.04300, 0.04100, 0.04100,
     0.04280, 0.03970, 0.04500, 0.04230, 0.03780, 0.04120, 0.04000, 0.04140, 0.04030, 0.04220,
     0.03780, 0.03540, 0.03970, 0.03670, 0.04050, 0.03740, 0.03880, 0.03510, 0.03860, 0.03490,
     0.03820, 0.03250, 0.03850, 0.03750, 0.03710, 0.03640, 0.03000, 0.03320, 0.03290, 0.03170,
     0.02880, 0.03370, 0.03020, 0.03250, 0.03290, 0.03140, 0.03420, 0.03050, 0.02780, 0.02830,
     0.03560, 0.03070, 0.02950, 0.02780, 0.03010, 0.03060, 0.02840, 0.03620, 0.02900, 0.03010,
     0.02980, 0.02500, 0.02560, 0.02710, 0.02630, 0.02590, 0.02470, 0.02700, 0.02590, 0.02810,
     0.02540, 0.02930, 0.02810, 0.02330, 0.02410, 0.02160, 0.02230, 0.02160, 0.02800, 0.02740,
     0.02200, 0.02020, 0.01870, 0.01770, 0.02580, 0.01950, 0.02050, 0.02040, 0.02220, 0.01950,
     0.02450, 0.02310, 0.01840, 0.02050, 0.01930, 0.01700, 0.01940, 0.01710, 0.01690},

    {0.04361, 0.04124, 0.04106, 0.04479, 0.04597, 0.05028, 0.04524, 0.04912, 0.05037, 0.05153,
     0.05215, 0.05286, 0.05340, 0.05437, 0.05842, 0.06164, 0.05877, 0.06312, 0.06363, 0.06294,
     0.05903, 0.06510, 0.06613, 0.06656, 0.07022, 0.06938, 0.07149, 0.07200, 0.07141, 0.07519,
     0.07234, 0.07711, 0.08001, 0.08157, 0.07744, 0.08174, 0.07960, 0.08581, 0.08370, 0.08345,
     0.08759, 0.08936, 0.09303, 0.09183, 0.09405, 0.09681, 0.10093, 0.09813, 0.09891, 0.10362,
     0.10408, 0.10162, 0.10877, 0.10930, 0.10967, 0.10780, 0.10923, 0.11251, 0.10795, 0.11516,
     0.11685, 0.11692, 0.12312, 0.12291, 0.12212, 0.12917, 0.12162, 0.13015, 0.13008, 0.13077,
     0.12904, 0.13616, 0.13331, 0.13454, 0.13582, 0.13984, 0.14567, 0.14366, 0.14248, 0.14554,
     0.14875, 0.14754, 0.15103, 0.15122, 0.15166, 0.15551, 0.15557, 0.16197, 0.15970, 0.15928,
     0.16380, 0.16561, 0.16636, 0.16929, 0.16889, 0.16677, 0.17261, 0.17149, 0.17705, 0.17382,
     0.17988, 0.17860, 0.18115, 0.18287, 0.18271, 0.18395, 0.18359, 0.18323, 0.18390, 0.19274,
     0.19269, 0.18948, 0.19585, 0.19226, 0.19478, 0.19543, 0.19655, 0.20278, 0.20420, 0.20213,
     0.20382, 0.20628, 0.20777, 0.20632, 0.20818, 0.21000, 0.20756, 0.21221, 0.21466, 0.21555,
     0.21466, 0.21636, 0.21533, 0.21731, 0.22127, 0.22147, 0.22130, 0.22513, 0.22231, 0.22510,
     0.22548, 0.22616, 0.22874, 0.22929, 0.22848, 0.22932, 0.23032, 0.23040, 0.23196, 0.23336,
     0.23294, 0.23728, 0.23619, 0.23615, 0.23941, 0.23657, 0.23761, 0.23876, 0.23876, 0.23947,
     0.24113, 0.23910, 0.24266, 0.24333, 0.24197, 0.24299, 0.24250, 0.24458, 0.24497, 0.24513,
     0.24597, 0.24658, 0.24621, 0.24647, 0.24717, 0.24572, 0.24808, 0.24776, 0.24805, 0.24822,
     0.24892, 0.24924, 0.24961, 0.24888, 0.24861, 0.24894, 0.24961, 0.24991, 0.24986, 0.24999,
     0.24985, 0.24996, 0.25000, 0.25000, 0.24995, 0.24992, 0.25000, 0.24998, 0.24965, 0.24981,
     0.24950, 0.24931, 0.24918, 0.24898, 0.24880, 0.24932, 0.24757, 0.24886, 0.24836, 0.24925,
     0.24803, 0.24768, 0.24707, 0.24577, 0.24712, 0.24763, 0.24616, 0.24496, 0.24547, 0.24458,
     0.24436, 0.24470, 0.24346, 0.24217, 0.24339, 0.24213, 0.24208, 0.24034, 0.24148, 0.23914,
     0.24065, 0.24022, 0.24051, 0.23799, 0.23902, 0.23650, 0.23600, 0.23445, 0.23687, 0.23395,
     0.23415, 0.23470, 0.23302, 0.23487, 0.23023, 0.23148, 0.23123, 0.22877, 0.22877, 0.22687,
     0.22653, 0.22705, 0.22582, 0.22588, 0.22430, 0.21941, 0.22414, 0.22469, 0.22427, 0.22124,
     0.21937, 0.22011, 0.21658, 0.21824, 0.22117, 0.21860, 0.21552, 0.21413, 0.21496, 0.21544,
     0.21317, 0.20887, 0.21166, 0.20781, 0.20561, 0.20399, 0.20686, 0.20842, 0.20326, 0.20446,
     0.20208, 0.20200, 0.20330, 0.20045, 0.19956, 0.20243, 0.19710, 0.19797, 0.19897, 0.19524,
     0.19543, 0.19506, 0.19081, 0.19211, 0.19378, 0.19302, 0.18594, 0.18619, 0.18614, 0.18405,
     0.18421, 0.18467, 0.18354, 0.17903, 0.18574, 0.17591, 0.18188, 0.17596, 0.17650, 0.17449,
     0.17432, 0.17238, 0.17569, 0.17299, 0.16883, 0.16923, 0.17272, 0.16951, 0.16613, 0.16729,
     0.16991, 0.16280, 0.16421, 0.15982, 0.16048, 0.15807, 0.15964, 0.16036, 0.15801, 0.15365,
     0.15752, 0.16126, 0.15291, 0.15892, 0.15259, 0.15316, 0.15477, 0.15059, 0.14773, 0.14359,
     0.14644, 0.14913, 0.14786, 0.14372, 0.14786, 0.14314, 0.14255, 0.14017, 0.13971, 0.14183,
     0.13576, 0.14037, 0.13413, 0.13228, 0.13697, 0.13283, 0.12652, 0.13426, 0.12973, 0.13008,
     0.13049, 0.12680, 0.12827, 0.11997, 0.12680, 0.12319, 0.12785, 0.12476, 0.12155, 0.11809,
     0.12069, 0.11619, 0.11634, 0.11787, 0.11648, 0.11177, 0.11568, 0.11421, 0.12191, 0.11663,
     0.11012, 0.11303, 0.11095, 0.10734, 0.11191, 0.10749, 0.10817, 0.10765, 0.10742, 0.09899,
     0.10537, 0.10362, 0.10392, 0.10392, 0.10201, 0.10193, 0.10514, 0.10231, 0.09539, 0.09649,
     0.09673, 0.09774, 0.09547, 0.09492, 0.09539, 0.09032, 0.09626, 0.08992, 0.09571, 0.08856,
     0.09239, 0.09390, 0.08678, 0.08856, 0.08654, 0.09160, 0.08419, 0.08075, 0.08362, 0.08565,
     0.08848, 0.08386, 0.08100, 0.07553, 0.07976, 0.07935, 0.08386, 0.07511, 0.07984, 0.07477,
     0.07786, 0.07335, 0.07519, 0.07039, 0.07293, 0.07544, 0.06895, 0.07149, 0.07200, 0.07394,
     0.07141, 0.07082, 0.07082, 0.07158, 0.07293, 0.06946, 0.07031, 0.06903, 0.06818, 0.06613,
     0.06562, 0.06458, 0.06441, 0.06346, 0.06320, 0.06199, 0.06690, 0.06190, 0.06121, 0.06372,
     0.05982, 0.05947, 0.06017, 0.05842, 0.06069, 0.05947, 0.06104, 0.05745, 0.05798, 0.05903,
     0.05955, 0.05781, 0.05534, 0.05921, 0.05490, 0.05789, 0.05357, 0.04885, 0.05366, 0.05153,
     0.04885, 0.05135, 0.05393, 0.04939, 0.05153, 0.04983, 0.04705, 0.04858, 0.04479, 0.05091,
     0.04642, 0.04379, 0.04750, 0.04606, 0.04579, 0.04316, 0.04633, 0.04325, 0.04714, 0.04124,
     0.04524, 0.03858, 0.04416, 0.04179, 0.04133, 0.04051, 0.04570, 0.04115, 0.03932, 0.03932,
     0.04097, 0.03812, 0.04297, 0.04051, 0.03637, 0.03950, 0.03840, 0.03969, 0.03868, 0.04042,
     0.03637, 0.03415, 0.03812, 0.03535, 0.03886, 0.03600, 0.03730, 0.03387, 0.03711, 0.03368,
     0.03674, 0.03144, 0.03702, 0.03609, 0.03572, 0.03508, 0.02910, 0.03210, 0.03182, 0.03070,
     0.02797, 0.03256, 0.02929, 0.03144, 0.03182, 0.03041, 0.03303, 0.02957, 0.02703, 0.02750,
     0.03433, 0.02976, 0.02863, 0.02703, 0.02919, 0.02966, 0.02759, 0.03489, 0.02816, 0.02919,
     0.02891, 0.02438, 0.02494, 0.02637, 0.02561, 0.02523, 0.02409, 0.02627, 0.02523, 0.02731,
     0.02475, 0.02844, 0.02731, 0.02276, 0.02352, 0.02113, 0.02180, 0.02113, 0.02722, 0.02665,
     0.02152, 0.01979, 0.01835, 0.01739, 0.02513, 0.01912, 0.02008, 0.01998, 0.02171, 0.01912,
     0.02390, 0.02257, 0.01806, 0.02008, 0.01893, 0.01671, 0.01902, 0.01681, 0.01661},

    {0.95021, 0.95292, 0.95313, 0.94885, 0.94750, 0.94251, 0.94833, 0.94386, 0.94240, 0.94105,
     0.94032, 0.93949, 0.93887, 0.93773, 0.93296, 0.92913, 0.93255, 0.92738, 0.92676, 0.92758,
     0.93224, 0.92500, 0.92376, 0.92324, 0.91881, 0.91984, 0.91726, 0.91664, 0.91736, 0.91273,
     0.91623, 0.91036, 0.90676, 0.90480, 0.90995, 0.90460, 0.90727, 0.89946, 0.90213, 0.90244,
     0.89720, 0.89494, 0.89022, 0.89176, 0.88889, 0.88530, 0.87987, 0.88356, 0.88254, 0.87629,
     0.87568, 0.87895, 0.86934, 0.86862, 0.86811, 0.87067, 0.86872, 0.86423, 0.87046, 0.86055,
     0.85820, 0.85810, 0.84932, 0.84963, 0.85075, 0.84056, 0.85147, 0.83913, 0.83923, 0.83821,
     0.84076, 0.83017, 0.83444, 0.83261, 0.83068, 0.82457, 0.81552, 0.81867, 0.82050, 0.81572,
     0.81064, 0.81257, 0.80698, 0.80668, 0.80597, 0.79967, 0.79957, 0.78881, 0.79267, 0.79338,
     0.78567, 0.78252, 0.78121, 0.77604, 0.77675, 0.78050, 0.77006, 0.77208, 0.76185, 0.76783,
     0.75649, 0.75892, 0.75406, 0.75072, 0.75102, 0.74859, 0.74930, 0.75001, 0.74869, 0.73069,
     0.73080, 0.73747, 0.72403, 0.73171, 0.72635, 0.72493, 0.72251, 0.70847, 0.70514, 0.70999,
     0.70605, 0.70020, 0.69657, 0.70010, 0.69556, 0.69102, 0.69707, 0.68537, 0.67892, 0.67650,
     0.67892, 0.67428, 0.67710, 0.67166, 0.66028, 0.65968, 0.66018, 0.64840, 0.65716, 0.64850,
     0.64729, 0.64508, 0.63643, 0.63452, 0.63733, 0.63441, 0.63089, 0.63059, 0.62486, 0.61953,
     0.62114, 0.60325, 0.60797, 0.60818, 0.59331, 0.60637, 0.60175, 0.59642, 0.59642, 0.59301,
     0.58457, 0.59482, 0.57605, 0.57203, 0.57996, 0.57404, 0.57695, 0.56391, 0.56120, 0.56010,
     0.55378, 0.54877, 0.55188, 0.54967, 0.54346, 0.55568, 0.53404, 0.53754, 0.53434, 0.53243,
     0.52302, 0.51782, 0.51001, 0.52362, 0.52753, 0.52282, 0.50991, 0.49990, 0.50200, 0.49410,
     0.50260, 0.49610, 0.49160, 0.49160, 0.48310, 0.48140, 0.48900, 0.48600, 0.47161, 0.47640,
     0.46781, 0.46391, 0.46152, 0.45822, 0.45552, 0.46421, 0.44095, 0.45642, 0.44973, 0.46281,
     0.44584, 0.44205, 0.43616, 0.42528, 0.43656, 0.44155, 0.42828, 0.41930, 0.42299, 0.41671,
     0.41521, 0.41750, 0.40943, 0.40186, 0.40903, 0.40165, 0.40136, 0.39209, 0.39807, 0.38621,
     0.39369, 0.39149, 0.39299, 0.38084, 0.38562, 0.37427, 0.37218, 0.36581, 0.37586, 0.36382,
     0.36462, 0.36680, 0.36024, 0.36750, 0.35000, 0.35447, 0.35357, 0.34492, 0.34492, 0.33856,
     0.33747, 0.33916, 0.33519, 0.33538, 0.33042, 0.31592, 0.32992, 0.33161, 0.33032, 0.32118,
     0.31582, 0.31790, 0.30808, 0.31264, 0.32098, 0.31364, 0.30520, 0.30153, 0.30371, 0.30500,
     0.29905, 0.28824, 0.29518, 0.28567, 0.28041, 0.27665, 0.28339, 0.28715, 0.27496, 0.27774,
     0.27229, 0.27209, 0.27506, 0.26863, 0.26664, 0.27308, 0.26130, 0.26318, 0.26536, 0.25734,
     0.25773, 0.25694, 0.24814, 0.25081, 0.25427, 0.25269, 0.23845, 0.23894, 0.23884, 0.23479,
     0.23509, 0.23598, 0.23380, 0.22531, 0.23805, 0.21958, 0.23064, 0.21968, 0.22067, 0.21701,
     0.21672, 0.21326, 0.21918, 0.21435, 0.20705, 0.20774, 0.21385, 0.20823, 0.20241, 0.20438,
     0.20892, 0.19679, 0.19916, 0.19186, 0.19295, 0.18901, 0.19157, 0.19275, 0.18891, 0.18192,
     0.18812, 0.19423, 0.18074, 0.19039, 0.18024, 0.18113, 0.18369, 0.17709, 0.17267, 0.16637,
     0.17070, 0.17483, 0.17286, 0.16657, 0.17286, 0.16568, 0.16480, 0.16126, 0.16057, 0.16372,
     0.15478, 0.16156, 0.15242, 0.14977, 0.15655, 0.15056, 0.14163, 0.15262, 0.14614, 0.14663,
     0.14722, 0.14202, 0.14408, 0.13261, 0.14202, 0.13702, 0.14349, 0.13918, 0.13477, 0.13006,
     0.13359, 0.12752, 0.12771, 0.12977, 0.12791, 0.12165, 0.12683, 0.12488, 0.13526, 0.12811,
     0.11950, 0.12331, 0.12057, 0.11588, 0.12184, 0.11607, 0.11695, 0.11627, 0.11598, 0.10523,
     0.11334, 0.11109, 0.11148, 0.11148, 0.10904, 0.10894, 0.11305, 0.10943, 0.10075, 0.10211,
     0.10240, 0.10367, 0.10084, 0.10016, 0.10075, 0.09451, 0.10182, 0.09402, 0.10114, 0.09237,
     0.09704, 0.09889, 0.09023, 0.09237, 0.08993, 0.09607, 0.08711, 0.08303, 0.08643, 0.08886,
     0.09227, 0.08672, 0.08332, 0.07691, 0.08186, 0.08138, 0.08672, 0.07643, 0.08196, 0.07604,
     0.07963, 0.07439, 0.07652, 0.07100, 0.07391, 0.07682, 0.06935, 0.07226, 0.07284, 0.07507,
     0.07216, 0.07148, 0.07148, 0.07236, 0.07391, 0.06993, 0.07090, 0.06945, 0.06848, 0.06616,
     0.06558, 0.06442, 0.06423, 0.06316, 0.06287, 0.06152, 0.06703, 0.06142, 0.06065, 0.06345,
     0.05911, 0.05872, 0.05949, 0.05756, 0.06007, 0.05872, 0.06046, 0.05650, 0.05708, 0.05824,
     0.05882, 0.05689, 0.05419, 0.05843, 0.05371, 0.05698, 0.05226, 0.04717, 0.05236, 0.05005,
     0.04717, 0.04986, 0.05265, 0.04774, 0.05005, 0.04822, 0.04525, 0.04688, 0.04285, 0.04938,
     0.04458, 0.04180, 0.04573, 0.04419, 0.04391, 0.04113, 0.04448, 0.04122, 0.04535, 0.03912,
     0.04333, 0.03635, 0.04218, 0.03969, 0.03921, 0.03836, 0.04381, 0.03902, 0.03711, 0.03711,
     0.03883, 0.03587, 0.04094, 0.03836, 0.03406, 0.03730, 0.03616, 0.03750, 0.03644, 0.03826,
     0.03406, 0.03178, 0.03587, 0.03302, 0.03664, 0.03368, 0.03501, 0.03149, 0.03482, 0.03130,
     0.03444, 0.02902, 0.03473, 0.03378, 0.03340, 0.03273, 0.02666, 0.02969, 0.02940, 0.02827,
     0.02552, 0.03016, 0.02685, 0.02902, 0.02940, 0.02798, 0.03064, 0.02713, 0.02458, 0.02505,
     0.03197, 0.02732, 0.02618, 0.02458, 0.02675, 0.02722, 0.02514, 0.03254, 0.02571, 0.02675,
     0.02647, 0.02194, 0.02250, 0.02392, 0.02316, 0.02279, 0.02166, 0.02382, 0.02279, 0.02486,
     0.02232, 0.02600, 0.02486, 0.02034, 0.02109, 0.01875, 0.01941, 0.01875, 0.02477, 0.02420,
     0.01912, 0.01744, 0.01605, 0.01512, 0.02269, 0.01679, 0.01772, 0.01763, 0.01931, 0.01679,
     0.02147, 0.02016, 0.01577, 0.01772, 0.01660, 0.01447, 0.01670, 0.01456, 0.01437},

    {0.95839, 0.96088, 0.96107, 0.95715, 0.95590, 0.95130, 0.95667, 0.95254, 0.95120, 0.94995,
     0.94928, 0.94851, 0.94793, 0.94687, 0.94244, 0.93887, 0.94205, 0.93722, 0.93664, 0.93742,
     0.94176, 0.93500, 0.93384, 0.93336, 0.92919, 0.93016, 0.92774, 0.92716, 0.92784, 0.92348,
     0.92677, 0.92124, 0.91784, 0.91600, 0.92085, 0.91580, 0.91833, 0.91094, 0.91347, 0.91376,
     0.90880, 0.90666, 0.90218, 0.90364, 0.90091, 0.89750, 0.89233, 0.89584, 0.89486, 0.88891,
     0.88832, 0.89145, 0.88226, 0.88158, 0.88109, 0.88353, 0.88168, 0.87737, 0.88334, 0.87385,
     0.87160, 0.87150, 0.86308, 0.86337, 0.86445, 0.85464, 0.86513, 0.85327, 0.85337, 0.85239,
     0.85484, 0.84463, 0.84876, 0.84699, 0.84512, 0.83923, 0.83048, 0.83353, 0.83530, 0.83068,
     0.82576, 0.82763, 0.82222, 0.82192, 0.82123, 0.81513, 0.81503, 0.80459, 0.80833, 0.80902,
     0.80153, 0.79848, 0.79719, 0.79216, 0.79285, 0.79650, 0.78634, 0.78832, 0.77835, 0.78417,
     0.77311, 0.77548, 0.77074, 0.76748, 0.76778, 0.76541, 0.76610, 0.76679, 0.76550, 0.74791,
     0.74800, 0.75453, 0.74137, 0.74889, 0.74365, 0.74226, 0.73989, 0.72613, 0.72286, 0.72761,
     0.72375, 0.71800, 0.71443, 0.71790, 0.71344, 0.70898, 0.71493, 0.70343, 0.69708, 0.69470,
     0.69708, 0.69252, 0.69529, 0.68994, 0.67872, 0.67812, 0.67862, 0.66700, 0.67564, 0.66710,
     0.66591, 0.66372, 0.65517, 0.65329, 0.65607, 0.65319, 0.64971, 0.64941, 0.64374, 0.63847,
     0.64006, 0.62235, 0.62703, 0.62723, 0.61249, 0.62543, 0.62085, 0.61558, 0.61558, 0.61219,
     0.60382, 0.61398, 0.59535, 0.59137, 0.59924, 0.59336, 0.59625, 0.58329, 0.58060, 0.57950,
     0.57322, 0.56823, 0.57132, 0.56913, 0.56294, 0.57512, 0.55356, 0.55706, 0.55386, 0.55197,
     0.54258, 0.53739, 0.52959, 0.54318, 0.54707, 0.54238, 0.52949, 0.51950, 0.52160, 0.51370,
     0.52220, 0.51570, 0.51120, 0.51120, 0.50270, 0.50100, 0.50860, 0.50560, 0.49119, 0.49600,
     0.48739, 0.48349, 0.48108, 0.47778, 0.47508, 0.48379, 0.46045, 0.47598, 0.46927, 0.48239,
     0.46536, 0.46155, 0.45564, 0.44472, 0.45604, 0.46105, 0.44772, 0.43870, 0.44241, 0.43609,
     0.43459, 0.43690, 0.42877, 0.42114, 0.42837, 0.42095, 0.42064, 0.41131, 0.41733, 0.40538,
     0.41291, 0.41071, 0.41221, 0.39996, 0.40478, 0.39333, 0.39122, 0.38479, 0.39494, 0.38278,
     0.38358, 0.38579, 0.37916, 0.38650, 0.36880, 0.37333, 0.37243, 0.36368, 0.36368, 0.35724,
     0.35613, 0.35784, 0.35381, 0.35402, 0.34898, 0.33428, 0.34848, 0.35019, 0.34888, 0.33962,
     0.33418, 0.33630, 0.32632, 0.33096, 0.33942, 0.33196, 0.32340, 0.31967, 0.32189, 0.32320,
     0.31715, 0.30616, 0.31322, 0.30353, 0.29819, 0.29435, 0.30121, 0.30505, 0.29264, 0.29546,
     0.28991, 0.28971, 0.29274, 0.28618, 0.28416, 0.29072, 0.27870, 0.28062, 0.28284, 0.27466,
     0.27507, 0.27426, 0.26526, 0.26799, 0.27153, 0.26991, 0.25535, 0.25586, 0.25576, 0.25161,
     0.25191, 0.25282, 0.25060, 0.24189, 0.25495, 0.23602, 0.24736, 0.23612, 0.23713, 0.23339,
     0.23308, 0.22954, 0.23561, 0.23065, 0.22315, 0.22386, 0.23015, 0.22437, 0.21839, 0.22042,
     0.22508, 0.21261, 0.21504, 0.20754, 0.20865, 0.20459, 0.20723, 0.20845, 0.20449, 0.19728,
     0.20368, 0.20997, 0.19606, 0.20601, 0.19556, 0.19647, 0.19911, 0.19231, 0.18773, 0.18123,
     0.18570, 0.18997, 0.18794, 0.18143, 0.18794, 0.18052, 0.17960, 0.17594, 0.17523, 0.17848,
     0.16922, 0.17624, 0.16678, 0.16403, 0.17105, 0.16484, 0.15557, 0.16698, 0.16026, 0.16077,
     0.16138, 0.15598, 0.15812, 0.14619, 0.15598, 0.15078, 0.15751, 0.15302, 0.14843, 0.14353,
     0.14721, 0.14088, 0.14108, 0.14323, 0.14129, 0.13475, 0.14017, 0.13812, 0.14894, 0.14149,
     0.13250, 0.13649, 0.13363, 0.12872, 0.13496, 0.12893, 0.12985, 0.12913, 0.12882, 0.11757,
     0.12606, 0.12371, 0.12412, 0.12412, 0.12156, 0.12146, 0.12576, 0.12197, 0.11285, 0.11429,
     0.11460, 0.11593, 0.11296, 0.11224, 0.11285, 0.10629, 0.11398, 0.10578, 0.11326, 0.10403,
     0.10896, 0.11091, 0.10177, 0.10403, 0.10147, 0.10793, 0.09849, 0.09417, 0.09777, 0.10034,
     0.10393, 0.09808, 0.09448, 0.08769, 0.09294, 0.09242, 0.09808, 0.08717, 0.09304, 0.08676,
     0.09057, 0.08501, 0.08728, 0.08140, 0.08449, 0.08758, 0.07965, 0.08274, 0.08336, 0.08573,
     0.08264, 0.08192, 0.08192, 0.08284, 0.08449, 0.08027, 0.08130, 0.07975, 0.07872, 0.07624,
     0.07562, 0.07438, 0.07417, 0.07304, 0.07273, 0.07128, 0.07717, 0.07118, 0.07035, 0.07335,
     0.06869, 0.06828, 0.06911, 0.06704, 0.06973, 0.06828, 0.07014, 0.06590, 0.06652, 0.06776,
     0.06838, 0.06631, 0.06341, 0.06797, 0.06289, 0.06642, 0.06134, 0.05583, 0.06144, 0.05895,
     0.05583, 0.05874, 0.06175, 0.05646, 0.05895, 0.05697, 0.05375, 0.05552, 0.05115, 0.05822,
     0.05302, 0.05000, 0.05427, 0.05261, 0.05229, 0.04927, 0.05292, 0.04938, 0.05386, 0.04708,
     0.05167, 0.04405, 0.05042, 0.04771, 0.04718, 0.04625, 0.05219, 0.04698, 0.04489, 0.04489,
     0.04677, 0.04353, 0.04906, 0.04625, 0.04154, 0.04510, 0.04384, 0.04530, 0.04415, 0.04614,
     0.04154, 0.03902, 0.04353, 0.04038, 0.04436, 0.04112, 0.04258, 0.03871, 0.04238, 0.03850,
     0.04196, 0.03598, 0.04227, 0.04122, 0.04081, 0.04007, 0.03334, 0.03671, 0.03640, 0.03513,
     0.03208, 0.03724, 0.03355, 0.03598, 0.03640, 0.03482, 0.03776, 0.03387, 0.03102, 0.03155,
     0.03923, 0.03408, 0.03282, 0.03102, 0.03345, 0.03398, 0.03166, 0.03986, 0.03229, 0.03345,
     0.03313, 0.02806, 0.02870, 0.03028, 0.02944, 0.02901, 0.02774, 0.03018, 0.02901, 0.03134,
     0.02848, 0.03261, 0.03134, 0.02626, 0.02711, 0.02445, 0.02519, 0.02445, 0.03123, 0.03060,
     0.02488, 0.02296, 0.02135, 0.02028, 0.02891, 0.02221, 0.02328, 0.02317, 0.02509, 0.02221,
     0.02753, 0.02604, 0.02103, 0.02328, 0.02200, 0.01953, 0.02210, 0.01964, 0.01943},
};

/** @}*/
} // namespace ns3

#endif /* WIMAX_DEFAULT_TRACES_H */
