/*
 * Copyright (C) 2019 Intel Corporation.  All rights reserved.
 * SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
 */

package main

import (
    "gitlab.alipay-inc.com/TNT_Runtime/ant-runtime/bindings/go/wamr"
    "fmt"
)

var wasmBytes = []byte {
    0x00, 0x61, 0x73, 0x6D, 0x01, 0x00, 0x00, 0x00, 0x01, 0x29, 0x07, 0x60,
    0x01, 0x7F, 0x01, 0x7F, 0x60, 0x02, 0x7F, 0x7F, 0x01, 0x7F, 0x60, 0x01,
    0x7F, 0x00, 0x60, 0x04, 0x7F, 0x7E, 0x7D, 0x7C, 0x00, 0x60, 0x02, 0x7E,
    0x7E, 0x01, 0x7E, 0x60, 0x02, 0x7C, 0x7F, 0x01, 0x7D, 0x60, 0x02, 0x7D,
    0x7C, 0x01, 0x7C, 0x02, 0x31, 0x04, 0x03, 0x65, 0x6E, 0x76, 0x06, 0x70,
    0x72, 0x69, 0x6E, 0x74, 0x66, 0x00, 0x01, 0x03, 0x65, 0x6E, 0x76, 0x04,
    0x70, 0x75, 0x74, 0x73, 0x00, 0x00, 0x03, 0x65, 0x6E, 0x76, 0x06, 0x6D,
    0x61, 0x6C, 0x6C, 0x6F, 0x63, 0x00, 0x00, 0x03, 0x65, 0x6E, 0x76, 0x04,
    0x66, 0x72, 0x65, 0x65, 0x00, 0x02, 0x03, 0x07, 0x06, 0x00, 0x03, 0x04,
    0x06, 0x05, 0x01, 0x05, 0x03, 0x01, 0x00, 0x01, 0x06, 0x13, 0x03, 0x7F,
    0x01, 0x41, 0x90, 0x29, 0x0B, 0x7F, 0x00, 0x41, 0x90, 0x09, 0x0B, 0x7F,
    0x00, 0x41, 0x90, 0x29, 0x0B, 0x07, 0x5F, 0x09, 0x06, 0x6D, 0x65, 0x6D,
    0x6F, 0x72, 0x79, 0x02, 0x00, 0x04, 0x66, 0x69, 0x62, 0x32, 0x00, 0x04,
    0x05, 0x74, 0x65, 0x73, 0x74, 0x31, 0x00, 0x05, 0x05, 0x74, 0x65, 0x73,
    0x74, 0x32, 0x00, 0x06, 0x05, 0x74, 0x65, 0x73, 0x74, 0x33, 0x00, 0x07,
    0x05, 0x74, 0x65, 0x73, 0x74, 0x34, 0x00, 0x08, 0x10, 0x5F, 0x5F, 0x6D,
    0x61, 0x69, 0x6E, 0x5F, 0x61, 0x72, 0x67, 0x63, 0x5F, 0x61, 0x72, 0x67,
    0x76, 0x00, 0x09, 0x0A, 0x5F, 0x5F, 0x64, 0x61, 0x74, 0x61, 0x5F, 0x65,
    0x6E, 0x64, 0x03, 0x01, 0x0B, 0x5F, 0x5F, 0x68, 0x65, 0x61, 0x70, 0x5F,
    0x62, 0x61, 0x73, 0x65, 0x03, 0x02, 0x0A, 0xA5, 0x03, 0x06, 0x37, 0x01,
    0x01, 0x7F, 0x41, 0x01, 0x21, 0x01, 0x20, 0x00, 0x41, 0x02, 0x4F, 0x04,
    0x7F, 0x41, 0x00, 0x21, 0x01, 0x03, 0x40, 0x20, 0x00, 0x41, 0x02, 0x6B,
    0x10, 0x04, 0x20, 0x01, 0x6A, 0x21, 0x01, 0x20, 0x00, 0x41, 0x01, 0x6B,
    0x22, 0x00, 0x41, 0x01, 0x4B, 0x0D, 0x00, 0x0B, 0x20, 0x01, 0x41, 0x01,
    0x6A, 0x05, 0x41, 0x01, 0x0B, 0x0B, 0x3F, 0x01, 0x01, 0x7F, 0x23, 0x00,
    0x41, 0x20, 0x6B, 0x22, 0x04, 0x24, 0x00, 0x20, 0x04, 0x41, 0x18, 0x6A,
    0x20, 0x03, 0x39, 0x03, 0x00, 0x20, 0x04, 0x41, 0x10, 0x6A, 0x20, 0x02,
    0xBB, 0x39, 0x03, 0x00, 0x20, 0x04, 0x20, 0x01, 0x37, 0x03, 0x08, 0x20,
    0x04, 0x20, 0x00, 0x36, 0x02, 0x00, 0x41, 0xD0, 0x08, 0x20, 0x04, 0x10,
    0x00, 0x1A, 0x20, 0x04, 0x41, 0x20, 0x6A, 0x24, 0x00, 0x0B, 0x3B, 0x01,
    0x01, 0x7F, 0x23, 0x00, 0x41, 0x20, 0x6B, 0x22, 0x02, 0x24, 0x00, 0x20,
    0x02, 0x20, 0x00, 0x37, 0x03, 0x00, 0x20, 0x02, 0x20, 0x01, 0x37, 0x03,
    0x08, 0x20, 0x02, 0x41, 0x10, 0x6A, 0x20, 0x00, 0x20, 0x01, 0x7C, 0x22,
    0x00, 0x37, 0x03, 0x00, 0x41, 0xF6, 0x08, 0x20, 0x02, 0x10, 0x00, 0x1A,
    0x20, 0x02, 0x41, 0x20, 0x6A, 0x24, 0x00, 0x20, 0x00, 0x0B, 0x40, 0x02,
    0x01, 0x7F, 0x01, 0x7C, 0x23, 0x00, 0x41, 0x20, 0x6B, 0x22, 0x02, 0x24,
    0x00, 0x20, 0x02, 0x20, 0x01, 0x39, 0x03, 0x08, 0x20, 0x02, 0x20, 0x00,
    0xBB, 0x22, 0x03, 0x39, 0x03, 0x00, 0x20, 0x02, 0x41, 0x10, 0x6A, 0x20,
    0x03, 0x20, 0x01, 0xA2, 0x22, 0x01, 0x39, 0x03, 0x00, 0x41, 0xB4, 0x08,
    0x20, 0x02, 0x10, 0x00, 0x1A, 0x20, 0x02, 0x41, 0x20, 0x6A, 0x24, 0x00,
    0x20, 0x01, 0x0B, 0x3D, 0x01, 0x01, 0x7F, 0x23, 0x00, 0x41, 0x20, 0x6B,
    0x22, 0x02, 0x24, 0x00, 0x20, 0x02, 0x20, 0x00, 0x39, 0x03, 0x00, 0x20,
    0x02, 0x20, 0x01, 0x36, 0x02, 0x08, 0x20, 0x02, 0x41, 0x10, 0x6A, 0x20,
    0x00, 0x20, 0x01, 0xB7, 0xA3, 0x22, 0x00, 0x39, 0x03, 0x00, 0x41, 0xC2,
    0x08, 0x20, 0x02, 0x10, 0x00, 0x1A, 0x20, 0x02, 0x41, 0x20, 0x6A, 0x24,
    0x00, 0x20, 0x00, 0xB6, 0x0B, 0x70, 0x00, 0x23, 0x00, 0x41, 0x20, 0x6B,
    0x22, 0x00, 0x24, 0x00, 0x41, 0x9A, 0x08, 0x10, 0x01, 0x1A, 0x02, 0x7F,
    0x41, 0x80, 0x08, 0x10, 0x02, 0x22, 0x01, 0x45, 0x04, 0x40, 0x41, 0x88,
    0x08, 0x10, 0x01, 0x1A, 0x41, 0x7F, 0x0C, 0x01, 0x0B, 0x20, 0x00, 0x20,
    0x01, 0x36, 0x02, 0x10, 0x41, 0xA7, 0x08, 0x20, 0x00, 0x41, 0x10, 0x6A,
    0x10, 0x00, 0x1A, 0x20, 0x01, 0x41, 0x04, 0x6A, 0x41, 0x8E, 0x09, 0x2F,
    0x00, 0x00, 0x3B, 0x00, 0x00, 0x20, 0x01, 0x41, 0x8A, 0x09, 0x28, 0x00,
    0x00, 0x36, 0x00, 0x00, 0x20, 0x00, 0x20, 0x01, 0x36, 0x02, 0x00, 0x41,
    0x80, 0x08, 0x20, 0x00, 0x10, 0x00, 0x1A, 0x20, 0x01, 0x10, 0x03, 0x41,
    0x00, 0x0B, 0x20, 0x00, 0x41, 0x20, 0x6A, 0x24, 0x00, 0x0B, 0x0B, 0x97,
    0x01, 0x01, 0x00, 0x41, 0x80, 0x08, 0x0B, 0x8F, 0x01, 0x62, 0x75, 0x66,
    0x3A, 0x20, 0x25, 0x73, 0x00, 0x6D, 0x61, 0x6C, 0x6C, 0x6F, 0x63, 0x20,
    0x62, 0x75, 0x66, 0x20, 0x66, 0x61, 0x69, 0x6C, 0x65, 0x64, 0x00, 0x48,
    0x65, 0x6C, 0x6C, 0x6F, 0x20, 0x77, 0x6F, 0x72, 0x6C, 0x64, 0x21, 0x00,
    0x62, 0x75, 0x66, 0x20, 0x70, 0x74, 0x72, 0x3A, 0x20, 0x25, 0x70, 0x0A,
    0x00, 0x25, 0x66, 0x20, 0x2A, 0x20, 0x25, 0x66, 0x20, 0x3D, 0x20, 0x25,
    0x66, 0x0A, 0x00, 0x25, 0x66, 0x20, 0x2F, 0x20, 0x25, 0x64, 0x20, 0x3D,
    0x20, 0x25, 0x66, 0x0A, 0x00, 0x69, 0x33, 0x32, 0x3A, 0x20, 0x25, 0x64,
    0x2C, 0x20, 0x69, 0x36, 0x34, 0x3A, 0x20, 0x25, 0x6C, 0x6C, 0x64, 0x2C,
    0x20, 0x66, 0x33, 0x32, 0x3A, 0x20, 0x25, 0x66, 0x2C, 0x20, 0x66, 0x36,
    0x34, 0x3A, 0x20, 0x25, 0x66, 0x0A, 0x00, 0x25, 0x6C, 0x6C, 0x64, 0x20,
    0x2B, 0x20, 0x25, 0x6C, 0x6C, 0x64, 0x20, 0x3D, 0x20, 0x25, 0x6C, 0x6C,
    0x64, 0x0A, 0x00, 0x31, 0x32, 0x33, 0x34, 0x0A }

var global_heap []byte = make([]byte, 128 * 1024, 128 * 1024)

func main() {
    var module *wamr.Module
    var instance *wamr.Instance
    var argv []uint32
    var results []interface{}
    var offset uint32
    var native_addr *uint8
    var err error

    fmt.Print("Init wasm runtime with global heap buf\n");
    err = wamr.Runtime().FullInit(true, global_heap, 1)
    if err != nil {
        return
    }
    fmt.Print("Destroy runtime\n");
    wamr.Runtime().Destroy()

    fmt.Print("Init wasm runtime without global heap buf\n");
    err = wamr.Runtime().FullInit(false, nil, 1)
    if err != nil {
        return
    }

    wamr.Runtime().SetLogLevel(wamr.LOG_LEVEL_WARNING)

    fmt.Print("Load wasm module\n");
    module, err = wamr.NewModule(wasmBytes)
    if err != nil {
        fmt.Println(err)
        goto fail
    }

    fmt.Print("Instantiate wasm module\n");
    instance, err = wamr.NewInstance(module, 16384, 16384)
    if err != nil {
        fmt.Println(err)
        goto fail
    }

    results = make([]interface{}, 8, 8)
    argv = make([]uint32, 8)

    fmt.Print("\nCall func __main_argc_argv with CallFunc:\n");
    err = instance.CallFunc("__main_argc_argv", 2, argv)
    if err != nil {
        fmt.Println(err)
        goto fail
    }

    fmt.Print("\nCall func __main_argc_argv with CallFuncV:\n");
    err = instance.CallFuncV("__main_argc_argv", 2, results,
                             (int32)(0), (int32)(0))
    if err != nil {
        fmt.Println(err)
        goto fail
    }

    fmt.Print("\nCall func `i32 fib2(i32)` with CallFunc:\n");
    argv[0] = 32
    err = instance.CallFunc("fib2", 1, argv)
    if err != nil {
        fmt.Println(err)
        goto fail
    }
    fmt.Printf("fib2(32) return: %d\n", argv[0]);

    fmt.Print("\nCall func `void test1(i32, i64, f32, f64)` with CallFuncV:\n");
    err = instance.CallFuncV("test1", 0, nil,
                             (int32)(12345678),
                             (int64)(3344556677889900),
                             (float32)(5678.1234),
                             (float64)(987654321.5678))
    if err != nil {
        fmt.Println(err)
        goto fail
    }

    fmt.Print("\nCall func `i64 test2(i64, i64)` with CallFuncV:\n");
    err = instance.CallFuncV("test2", 1, results,
                             (int64)(3344556677889900),
                             (int64)(1122331122110099))
    if err != nil {
        fmt.Println(err)
        goto fail
    }
    fmt.Printf("test2(3344556677889900, 1122331122110099) return: %d\n",
               results[0].(int64))

    fmt.Print("\nCall func `f64 test3(f32, f64)` with CallFuncV:\n");
    err = instance.CallFuncV("test3", 1, results,
                             (float32)(3456.1234),
                             (float64)(7890.4567))
    if err != nil {
        fmt.Println(err)
        goto fail
    }
    fmt.Printf("test3(3456.1234, 7890.4567) return: %f\n",
               results[0].(float64))

    fmt.Print("\nCall func `f32 test4(f64, i32)` with CallFuncV:\n");
    err = instance.CallFuncV("test4", 1, results,
                             (float64)(8912.3456),
                             (int32)(123))
    if err != nil {
        fmt.Println(err)
        goto fail
    }
    fmt.Printf("test4(8912.3456, 123) return: %f\n",
               results[0].(float32))

    fmt.Print("\nTest ModuleMalloc")
    offset, native_addr = instance.ModuleMalloc(1024)
    fmt.Printf("ModuleMalloc(%d) return offset: %d, native addr: %p\n",
               1024, offset, native_addr)

    if (!instance.ValidateAppAddr(offset, 1024)) {
        fmt.Print("Validate app addr failed\n")
    }
    if (!instance.ValidateNativeAddr(native_addr, 1024)) {
        fmt.Print("Validate native addr failed\n")
    }
    if (native_addr != instance.AddrAppToNative(offset)) {
        fmt.Print("Convert app addr to native addr failed\n")
    }
    if (offset != instance.AddrNativeToApp(native_addr)) {
        fmt.Print("Convert app addr to native addr failed\n")
    }

    instance.ModuleFree(offset)

    /*
    instance.DumpMemoryConsumption()
    instance.DumpCallStack()
    */

    fmt.Print("\n");

fail:
    if (instance != nil) {
        fmt.Print("Destroy instance\n");
        instance.Destroy()
    }

    if (module != nil) {
        fmt.Print("Destroy module\n");
        module.Destroy()
    }

    fmt.Print("Destroy wasm runtime\n");
    wamr.Runtime().Destroy()
}
