/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.cnd.modeldiscovery.provider;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.netbeans.modules.cnd.api.model.CsmFile;
import org.netbeans.modules.cnd.api.model.CsmInclude;
import org.netbeans.modules.cnd.utils.CndPathUtilitities;
import org.netbeans.modules.cnd.discovery.api.ItemProperties;
import org.netbeans.modules.cnd.discovery.api.ItemProperties.LanguageKind;
import org.netbeans.modules.cnd.makeproject.spi.configurations.PkgConfigManager.PackageConfiguration;
import org.netbeans.modules.cnd.makeproject.spi.configurations.PkgConfigManager.PkgConfig;
import org.netbeans.modules.cnd.makeproject.spi.configurations.PkgConfigManager.ResolvedPath;
import org.netbeans.modules.cnd.discovery.api.SourceFileProperties;
import org.netbeans.modules.cnd.makeproject.api.configurations.Item;
import org.netbeans.modules.cnd.utils.cache.CndFileUtils;
import org.openide.util.Utilities;

/**
 *
 * @author Alexander Simon
 */
public class ModelSource implements SourceFileProperties {
    private static final boolean TRACE_AMBIGUOUS = Boolean.getBoolean("cnd.modeldiscovery.trace.ambiguous"); // NOI18N
    private static final Logger logger = Logger.getLogger("org.netbeans.modules.cnd.modeldiscovery.provider.SourceFileProperties"); // NOI18N
    {
        if (TRACE_AMBIGUOUS){logger.setLevel(Level.ALL);}
    }

    private Item item;
    private CsmFile file;
    private Map<String,List<String>> searchBase;
    private Map<String,Item> projectSearchBase;
    private PkgConfig pkgConfig;
    private String itemPath;
    private List<String> userIncludePaths;
    private Set<String> includedFiles = new HashSet<String>();
    private Map<String,String> userMacros;
    private boolean preferLocal;
    
    public ModelSource(Item item, CsmFile file, Map<String,List<String>> searchBase, Map<String,Item> projectSearchBase, PkgConfig pkgConfig, boolean preferLocal){
        this.item = item;
        this.file = file;
        this.searchBase = searchBase;
        this.projectSearchBase = projectSearchBase;
        this.pkgConfig = pkgConfig;
        this.preferLocal = preferLocal;
    }

    public Set<String> getIncludedFiles() {
        if (userIncludePaths == null) {
            getUserInludePaths();
        }
        return includedFiles;
    }

    @Override
    public String getCompilePath() {
        return new File( getItemPath()).getParentFile().getAbsolutePath();
    }
    
    @Override
    public String getItemPath() {
        if (itemPath == null) {
            itemPath = item.getAbsPath();
            itemPath = itemPath.replace('\\','/');
            itemPath = cutLocalRelative(itemPath);
            if (Utilities.isWindows()) {
                itemPath = itemPath.replace('/', File.separatorChar);
            }
        }
        return itemPath;
    }
    
    private static final String PATTERN = "/../"; // NOI18N
    public static String cutLocalRelative(String path){
        String pattern = PATTERN;
        while(true) {
            int i = path.indexOf(pattern);
            if (i < 0){
                break;
            }
            int k = -1;
            for (int j = i-1; j >= 0; j-- ){
                if ( path.charAt(j)=='/'){
                    k = j;
                    break;
                }
            }
            if (k<0) {
                break;
            }
            path = path.substring(0,k+1)+path.substring(i+pattern.length());
        }
        return path;
    }
    
    @Override
    public String getItemName() {
        return item.getName();
    }
    
    @Override
    public List<String> getUserInludePaths() {
        if (userIncludePaths == null) {
            List<String> includePaths = CndFileUtils.toPathList(item.getUserIncludePaths());
            Set<String> res = new LinkedHashSet<String>();
            for(String path : includePaths){
                path = getRelativepath(path);
                res.add(path);
            }
            analyzeUnresolved(res,file, 0);
            userIncludePaths = new ArrayList<String>(res);
        }
        return userIncludePaths;
    }
    
    private String getRelativepath(String path){
        if (Utilities.isWindows()) {
            path = path.replace('/', File.separatorChar);
        }
        path = CndPathUtilitities.toRelativePath(getCompilePath(), path);
        path = CndPathUtilitities.normalizeSlashes(path);
        return path;
    }
    
    private void analyzeUnresolved(Set<String> res, CsmFile what, int level){
        if (what == null) {
            return;
        }
        for (CsmInclude include : what.getIncludes()){
            CsmFile resolved = include.getIncludeFile();
            if (resolved == null){
                // unresolved include
                String path = guessPath(include);
                if (path != null) {
                    String fullPath = CndFileUtils.normalizeAbsolutePath(path + File.separatorChar + include.getIncludeName());
                    if (!file.getProject().isArtificial()) {
                        Item aItem = projectSearchBase.get(fullPath);
                        if (aItem != null) {
                            resolved = file.getProject().findFile(aItem, true, false);
                        }
                    } else {
                        resolved = file.getProject().findFile(fullPath, true, false);
                    }
                    path = getRelativepath(path);
                    res.add(path);
                    if (level < 5 && resolved != null) {
                        analyzeUnresolved(res, resolved, level+1);
                    }
                } else {
                    if (pkgConfig != null) {
                        Collection<ResolvedPath> listRP = pkgConfig.getResolvedPath(include.getIncludeName().toString());
                        if (listRP != null && !listRP.isEmpty()) {
                            ResolvedPath rp = listRP.iterator().next();
                            res.add(rp.getIncludePath());
                            for(PackageConfiguration pc : rp.getPackages()){
                                for(String p : pc.getIncludePaths()){
                                    if (!getSystemInludePaths().contains(p)){
                                        res.add(p);
                                    }
                                }
                                for(String p : pc.getMacros()){
                                    int i = p.indexOf('='); // NOI18N
                                    String macro;
                                    String value = null;
                                    if (i > 0){
                                        macro = p.substring(0, i);
                                        value = p.substring(i+1);
                                    } else {
                                        macro = p;
                                    }
                                    if (!getUserMacros().containsKey(macro)){
                                        getUserMacros().put(macro, value);
                                    }
                                }
                            }
                        }
                    }
                }
            } else {
                boolean reResolve = false;
                if (preferLocal) {
                    boolean isSystem = false;
                    String resolvedPath = resolved.getAbsolutePath().toString();
                    for (String path : getSystemInludePaths()){
                        if (resolvedPath.startsWith(path)) {
                            isSystem = true;
                            break;
                        }
                    }
                    if (isSystem) {
                        String path = guessPath(include);
                        if (path != null && !resolvedPath.startsWith(path)){
                            if (TRACE_AMBIGUOUS) {
                                logger.log(Level.FINE, "Directive resolved in project on path: {0} instead {1}", new Object[]{path, resolvedPath}); // NOI18N
                            }
                            String fullPath = CndFileUtils.normalizeAbsolutePath(path+File.separatorChar+include.getIncludeName());
                            resolved = file.getProject().findFile(fullPath, true, false);
                            path = getRelativepath(path);
                            res.add(path);
                            reResolve = true;
                        }
                    }
                }
                if (!reResolve) {
                    includedFiles.add(resolved.getAbsolutePath().toString());
                }
                if (level < 5 && resolved != null) {
                    analyzeUnresolved(res, resolved, level+1);
                }
            }
        }
    }
    
    private String guessPath(CsmInclude include){
        String name = include.getIncludeName().toString();
        String found = name.replace('\\','/');
        String prefix = null;
        //String back = null;
        int i = found.lastIndexOf('/');
        if(i >= 0){
            prefix = found.substring(0,i+1);
            found = found.substring(i+1);
            i = prefix.lastIndexOf("./"); // NOI18N
            if (i >= 0) {
                //back = prefix.substring(0,i+2);
                prefix = prefix.substring(i+2);
                if (prefix.length()==0) {
                    prefix = null;
                    name = found;
                } else {
                    name = prefix+'/'+found;
                }
            }
        }
        List<String> result = searchBase.get(found);
        if (result != null && result.size()>0){
            int pos = -1;
            //TODO: resolve ambiguously
            for(int j = 0; j < result.size(); j++){
                if (result.get(j).endsWith(name)){
                    if (pos >= 0) {
                        if (TRACE_AMBIGUOUS) {
                            logger.log(Level.FINE, "Ambiguous name for item: {0}", getItemPath()); // NOI18N
                            logger.log(Level.FINE, "  name1: {0}", result.get(pos)); // NOI18N
                            logger.log(Level.FINE, "  name2: {0}", result.get(j)); // NOI18N
                        }
                    } else {
                        pos = j;
                    }
                }
            }
            if (pos >=0) {
                String path = result.get(pos);
                path = path.substring(0,path.length()-name.length()-1);
                return path;
            }
        }
        if (TRACE_AMBIGUOUS) {
            logger.log(Level.FINE, "Unresolved name for item: {0}", getItemPath()); // NOI18N
            logger.log(Level.FINE, "  from: {0}", include.getContainingFile().getAbsolutePath()); // NOI18N
            logger.log(Level.FINE, "  name: {0}", include.getIncludeName()); // NOI18N
            if (result != null && result.size()>0){
                for(int j = 0; j < result.size(); j++){
                    logger.log(Level.FINE, "  candidate: {0}", result.get(j)); // NOI18N
                }
            }
        }
        return null;
    }
    
    @Override
    public List<String> getSystemInludePaths() {
        return CndFileUtils.toPathList(item.getSystemIncludePaths());
    }
    
    @Override
    public Map<String, String> getUserMacros() {
        if (userMacros == null){
            userMacros = new HashMap<String,String>();
            List<String> macros = item.getUserMacroDefinitions();
            for(String macro : macros){
                int i = macro.indexOf('=');
                if (i>0){
                    userMacros.put(macro.substring(0,i).trim(),macro.substring(i+1).trim());
                } else {
                    userMacros.put(macro,null);
                }
            }
        }
        return userMacros;
    }
    
    @Override
    public Map<String, String> getSystemMacros() {
        return null;
    }
    
    @Override
    public ItemProperties.LanguageKind getLanguageKind() {
        switch(item.getLanguage()) {
            case C:
                return LanguageKind.C;
            case CPP:
                return LanguageKind.CPP;
            case FORTRAN:
                return LanguageKind.Fortran;
        }
        return LanguageKind.Unknown;
    }

    @Override
    public String getCompilerName() {
        return null;
    }

    @Override
    public LanguageStandard getLanguageStandard() {
        return LanguageStandard.Unknown;
    }
}

