/* ClasspathManifestTest.java
 Copyright (C) 2012 Red Hat, Inc.

 This file is part of IcedTea.

 IcedTea is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License as published by
 the Free Software Foundation, version 2.

 IcedTea is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with IcedTea; see the file COPYING.  If not, write to
 the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 02110-1301 USA.

 Linking this library statically or dynamically with other modules is
 making a combined work based on this library.  Thus, the terms and
 conditions of the GNU General Public License cover the whole
 combination.

 As a special exception, the copyright holders of this library give you
 permission to link this library with independent modules to produce an
 executable, regardless of the license terms of these independent
 modules, and to copy and distribute the resulting executable under
 terms of your choice, provided that you also meet, for each linked
 independent module, the terms and conditions of the license of that
 module.  An independent module is a module which is not derived from
 or based on this library.  If you modify this library, you may extend
 this exception to your version of the library, but you are not
 obligated to do so.  If you do not wish to do so, delete this
 exception statement from your version.
 */

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import net.sourceforge.jnlp.ProcessResult;
import net.sourceforge.jnlp.ServerAccess;
import net.sourceforge.jnlp.annotations.NeedsDisplay;
import net.sourceforge.jnlp.annotations.TestInBrowsers;
import net.sourceforge.jnlp.browsertesting.BrowserTest;
import net.sourceforge.jnlp.browsertesting.Browsers;
import net.sourceforge.jnlp.browsertesting.browsers.firefox.FirefoxProfilesOperator;
import net.sourceforge.jnlp.util.FileUtils;
import org.junit.AfterClass;

import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;

public class ClasspathManifestTest extends BrowserTest {

    private static final String s1 = "Searching for CheckForClasspath.";
    private static final String s2 = "CheckForClasspath found on classpath.";
    private static final String ss = "xception";

    private static final String n1 = "ClasspathManifestJNLPHrefTest.html";
    private static final String n4 = "ClasspathManifestApplicationTest.jnlp";
    private static final String n2 = "ClasspathManifestAppletTest.jnlp";
    private static final String n3 = "ClasspathManifestAppletTest.html";
    private static final String[] ns = new String[]{n1, n2, n3, n4};
    private static final String n0 = "ClasspathManifestTest.jar";

    private static File newRoot;
    private static File newRoot1;

    public void checkAppFails(ProcessResult pr, String testName) {
        Assert.assertTrue("ClasspathManifest." + testName + " stdout should contain " + s1 + " but didn't", pr.stdout.contains(s1));
        Assert.assertFalse("ClasspathManifest." + testName + " stdout should not contain " + s2 + " but did", pr.stdout.contains(s2));
        Assert.assertTrue("ClasspathManifest." + testName + " stderr should contain " + ss + " but didn't", pr.stderr.contains(ss));
    }

    public void checkAppPass(ProcessResult pr, String testName) {
        Assert.assertTrue("ClasspathManifest." + testName + " stdout should contain " + s1 + " but didn't", pr.stdout.contains(s1));
        Assert.assertTrue("ClasspathManifest." + testName + " stdout should not contain " + s2 + " but did", pr.stdout.contains(s2));
        Assert.assertFalse("ClasspathManifest." + testName + " stderr should contain " + ss + " but didn't", pr.stderr.contains(ss));
    }

    @AfterClass
    public static void removeAlternativeLocalDirs() throws IOException {
        FirefoxProfilesOperator.deleteRecursively(newRoot);
    }

    @BeforeClass
    public static void createAlternativeLocalDirs() throws IOException {
        newRoot = File.createTempFile("itw", "ClasspathManifestTest");
        newRoot.delete();
        newRoot.mkdirs();
        newRoot.deleteOnExit();
        newRoot1 = new File(newRoot, "r1");
        newRoot1.mkdir();
        FirefoxProfilesOperator.copyRecursively(new File(server.getDir(), "Classpath"), newRoot);

        for (String n : ns) {
            copyTextFile(new File(server.getDir(), n), new File(newRoot, n));
        }
        FirefoxProfilesOperator.copyFile(new File(server.getDir(), n0), new File(newRoot1, n0));
    }

    public static void copyTextFile(File from, File to) throws IOException {
        String s = FileUtils.loadFileAsString(from);
//        for (String n : ns) {
//            s = s.replaceAll(n, newRoot1.getName()+"/" + n);
//        }
        s = s.replaceAll(n0, newRoot1.getName() + "/" + n0);
        FileUtils.saveFile(s, to);
    }

    @NeedsDisplay
    @Test
    public void ApplicationJNLPRemoteTest() throws Exception {
        ProcessResult pr = server.executeJavawsHeadless(null, "/" + n4);
        checkAppFails(pr, "ApplicationJNLPRemoteTest");
    }

    /**
     * See the difference between *LocalTest() and *LocalTest_differentDir().
     *
     * Itw always have "." on classpath. So
     *
     * ./jnlp or ./html (calling to jar.jar) + ./jar.jar +
     * ./Codebase/../second.jar are all on classapth but ./jnlp or ./html
     * (calling to someDir/jar.jar)+ ./someDir/jar.jar +
     * ./Codebase/../second.jar Is making the jar.jar laodable for startup, but
     * diapearing after encauntering Class-Path: in Manifest.mf
     *
     * @throws Exception
     */
    @NeedsDisplay
    @Test
    public void ApplicationJNLPLocalTest() throws Exception {
        List<String> commands = new ArrayList<>(3);
        commands.add(server.getJavawsLocation());
        commands.add(ServerAccess.HEADLES_OPTION);
        commands.add(n4);
        ProcessResult pr = ServerAccess.executeProcess(commands, server.getDir());
        checkAppPass(pr, "ApplicationJNLPLocalTest");
    }

    @NeedsDisplay
    @Test
    public void ApplicationJNLPLocalTest_differentDir() throws Exception {
        List<String> commands = new ArrayList<>(3);
        commands.add(server.getJavawsLocation());
        commands.add(ServerAccess.HEADLES_OPTION);
        commands.add(n4);
        ProcessResult pr = ServerAccess.executeProcess(commands, newRoot);
        checkAppFails(pr, "ApplicationJNLPLocalTest_differentDir");
    }

    @NeedsDisplay
    @Test
    public void AppletJNLPRemoteTest() throws Exception {
        ProcessResult pr = server.executeJavawsHeadless(null, "/" + n2);
        checkAppFails(pr, "AppletJNLPRemoteTest");
    }

    @NeedsDisplay
    @Test
    public void AppletJNLPRLocalTest() throws Exception {
        List<String> commands = new ArrayList<>(3);
        commands.add(server.getJavawsLocation());
        commands.add(ServerAccess.HEADLES_OPTION);
        commands.add(n2);
        ProcessResult pr = ServerAccess.executeProcess(commands, server.getDir());
        checkAppPass(pr, "AppletJNLPRLocalTest");
    }

    @NeedsDisplay
    @Test
    public void AppletJNLPRLocalTest_differentDir() throws Exception {
        List<String> commands = new ArrayList<>(3);
        commands.add(server.getJavawsLocation());
        commands.add(ServerAccess.HEADLES_OPTION);
        commands.add(n2);
        ProcessResult pr = ServerAccess.executeProcess(commands, newRoot);
        checkAppFails(pr, "AppletJNLPRLocalTest_differentDir");
    }

    @NeedsDisplay
    @TestInBrowsers(testIn = {Browsers.one})
    @Test
    public void BrowserJNLPHrefRemoteTest() throws Exception {
        ProcessResult pr = server.executeBrowser("/" + n1);
        checkAppFails(pr, "BrowserJNLPHrefRemoteTest");
    }

    @NeedsDisplay
    @TestInBrowsers(testIn = {Browsers.one})
    @Test
    public void BrowserJNLPHrefLocalTest() throws Exception {
        List<String> commands = new ArrayList<>(2);
        commands.add(server.getBrowserLocation());
        commands.add(n1);
        ProcessResult pr = ServerAccess.executeProcess(commands, server.getDir());
        checkAppPass(pr, "BrowserJNLPHrefLocalTest");
    }

    @NeedsDisplay
    @TestInBrowsers(testIn = {Browsers.one})
    @Test
    public void BrowserJNLPHrefLocalTest_differentDir() throws Exception {
        List<String> commands = new ArrayList<>(2);
        commands.add(server.getBrowserLocation());
        commands.add(n1);
        ProcessResult pr = ServerAccess.executeProcess(commands, newRoot);
        checkAppFails(pr, "BrowserJNLPHrefLocalTest_differentDir");
    }

    @NeedsDisplay
    @TestInBrowsers(testIn = {Browsers.one})
    @Test
    public void BrowserAppletRemoteTest() throws Exception {
        ProcessResult pr = server.executeBrowser("/" + n3);
        Assert.assertTrue("ClasspathManifest.BrowserAppletRemoteTest stdout should contain " + s1 + " but didn't", pr.stdout.contains(s1));
        // Should be the only one to search manifest for classpath.
        Assert.assertTrue("ClasspathManifest.BrowserAppletRemoteTest stdout should contain " + s2 + " but didn't", pr.stdout.contains(s2));
    }
}
