/*
 * Copyright (C) 2007-2012 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

static uint32_t
get_segment_base_and_check_limit(
	struct cpssp *cpssp,
	uint32_t addr,
	uint8_t seg,
	unsigned length,
	bool *serrp
)
{
	uint32_t segment_limit;
	uint32_t segment_base;

	switch (seg) {
	case SEGMENT_ES:
		segment_limit = cpssp->es_segment_limit;
		segment_base = cpssp->es_segment_base;
		break;
	case SEGMENT_CS:
		segment_limit = cpssp->cs_segment_limit;
		segment_base = cpssp->cs_segment_base;
		break;
	case SEGMENT_SS:
		segment_limit = cpssp->ss_segment_limit;
		segment_base = cpssp->ss_segment_base;
		break;
	case SEGMENT_DS:
		segment_limit = cpssp->ds_segment_limit;
		segment_base = cpssp->ds_segment_base;
		break;
	case SEGMENT_FS:
		segment_limit = cpssp->fs_segment_limit;
		segment_base = cpssp->fs_segment_base;
		break;
	case SEGMENT_GS:
		segment_limit = cpssp->gs_segment_limit;
		segment_base = cpssp->gs_segment_base;
		break;
	case SEGMENT_SS_NEW:
		segment_limit = cpssp->descriptor_segment_limit;
		segment_base = cpssp->descriptor_segment_base;
		break;
	case SEGMENT_GDT:
		segment_limit = cpssp->gdtr_limit;
		segment_base = cpssp->gdtr_base;
		break;
	case SEGMENT_LDT:
		segment_limit = cpssp->ldtr_system_limit;
		segment_base = cpssp->ldtr_system_base;
		break;
	case SEGMENT_IDT:
		segment_limit = cpssp->idtr_limit;
		segment_base = cpssp->idtr_base;
		break;
	case SEGMENT_TSS:
		segment_limit = cpssp->tr_system_limit;
		segment_base = cpssp->tr_system_base;
		break;
	default:
		assert(0); /* Mustn'nt happen. */
	}

	/* TODO: Add remaining checks */

	if (addr > segment_limit - length + 1) {
#if 0
		fprintf(stderr, "WARNING: #GP(0) address=0x%08x seg=%d "
				"length=%u segment_base=0x%08x "
				"segment_limt=0x%08x\n",
				addr, seg, length, segment_base,
				segment_limit);
		assert(0);
#endif
		if (! cpssp->cr0_pe) { /* FIXME */
			// *serrp = 1;
			// return 0;
		}
	}

	*serrp = 0; /* FIXME */
	return segment_base;
}

static bool
is_user(struct cpssp *cpssp, int seg)
{
	return cpssp->cs_dpl == 3
		&& (seg == SEGMENT_CS
		 || seg == SEGMENT_SS
		 || seg == SEGMENT_DS
		 || seg == SEGMENT_ES
		 || seg == SEGMENT_FS
		 || seg == SEGMENT_GS);
}

static void
NAME_(mrb)(
	struct cpssp *cpssp,
	uint8_t seg,
	uint32_t addr,
	uint8_t *valp,
	bool *serrp,
	uint8_t *merrp
)
{
	uint32_t base;

	base = get_segment_base_and_check_limit(cpssp, addr, seg, 1, serrp);
	if (*serrp) return;
	CHIP_(umrb)(cpssp, addr + base, is_user(cpssp, seg), valp, merrp);
}

static void
NAME_(mrw)(
	struct cpssp *cpssp,
	uint8_t seg,
	uint32_t addr,
	uint16_t *valp,
	bool *serrp,
	uint8_t *merrp
)
{
	uint32_t base;

	base = get_segment_base_and_check_limit(cpssp, addr, seg, 2, serrp);
	if (*serrp) return;
	CHIP_(umrw)(cpssp, addr + base, is_user(cpssp, seg), valp, merrp);
}

static void
NAME_(mrd)(
	struct cpssp *cpssp,
	uint8_t seg,
	uint32_t addr,
	uint32_t *valp,
	bool *serrp,
	uint8_t *merrp
)
{
	uint32_t base;

	base = get_segment_base_and_check_limit(cpssp, addr, seg, 4, serrp);
	if (*serrp) return;
	CHIP_(umrd)(cpssp, addr + base, is_user(cpssp, seg), valp, merrp);
}

static void
NAME_(mwb)(
	struct cpssp *cpssp,
	uint8_t seg,
	uint32_t addr,
	uint8_t val,
	bool *serrp,
	uint8_t *merrp
)
{
	uint32_t base;

	base = get_segment_base_and_check_limit(cpssp, addr, seg, 1, serrp);
	if (*serrp) return;
	CHIP_(umwb)(cpssp, addr + base, is_user(cpssp, seg), val, merrp);
}

static void
NAME_(mww)(
	struct cpssp *cpssp,
	uint8_t seg,
	uint32_t addr,
	uint16_t val,
	bool *serrp,
	uint8_t *merrp
)
{
	uint32_t base;

	base = get_segment_base_and_check_limit(cpssp, addr, seg, 2, serrp);
	if (*serrp) return;
	CHIP_(umww)(cpssp, addr + base, is_user(cpssp, seg), val, merrp);
}

static void
NAME_(mwd)(
	struct cpssp *cpssp,
	uint8_t seg,
	uint32_t addr,
	uint32_t val,
	bool *serrp,
	uint8_t *merrp
)
{
	uint32_t base;

	base = get_segment_base_and_check_limit(cpssp, addr, seg, 4, serrp);
	if (*serrp) return;
	CHIP_(umwd)(cpssp, addr + base, is_user(cpssp, seg), val, merrp);
}
