/*
*****************************************************************************
*
* Copyright (c) 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Teunis van Beelen
* All rights reserved.
*
* email: teuniz@gmail.com
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions are met:
*     * Redistributions of source code must retain the above copyright
*       notice, this list of conditions and the following disclaimer.
*     * Redistributions in binary form must reproduce the above copyright
*       notice, this list of conditions and the following disclaimer in the
*       documentation and/or other materials provided with the distribution.
*
* THIS SOFTWARE IS PROVIDED BY Teunis van Beelen ''AS IS'' AND ANY
* EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL Teunis van Beelen BE LIABLE FOR ANY
* DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
* ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
*****************************************************************************
*/

/*
  this program generates an EDFplus or BDFplus testfile with the following signals:

 signal label/waveform  amplitude    f         sf
 ----------------------------------------------------
    1    squarewave        100 uV    0.1Hz     200 Hz
    2    ramp              100 uV    1 Hz      200 Hz
    3    pulse             100 uV    1 Hz      200 Hz
    4    noise             100 uV    - Hz      200 Hz
    5    sine 1 Hz         100 uV    1 Hz      200 Hz
    6    sine 3.05 Hz      100 uV    3.05 Hz   200 Hz
    7    sine 5.025 Hz     100 uV    5.025 Hz  200 Hz
    8    sine 8 Hz         100 uV    8 Hz      200 Hz
    9    sine 8.25 Hz      100 uV    8.25 Hz   200 Hz
   10    sine 8.5 Hz       100 uV    8.5Hz     200 Hz
   11    sine 15 Hz        100 uV   15 Hz      200 Hz
   12    sine 17 Hz        100 uV   17 Hz      200 Hz
   13    sine 17.05 Hz     100 uV   17.05 Hz   200 Hz
   14    sine 17.025 Hz    100 uV   17.025 Hz  200 Hz
   15    sine 50 Hz        100 uV   50 Hz      200 Hz
   16    DC                100 uV    0 Hz      200 Hz
   17    DC slope              uV    0 Hz      200 Hz
   18    sine 2 Hz  0.5 uV AC + 500 uV DC   2 Hz    200 Hz
   19    sine 2 Hz  0.5 uV AC + 500 uV DC   2 Hz    200 Hz  (inverted)
   20    Impuls              3 mV    0.1 Hz   1000 Hz
   21    Triangular 20mS   1.5 mV    0.1 Hz   1000 Hz
   22    Triangular 200mS  1.5 mV    0.1 Hz   1000 Hz
   23    Spike 0.5mS         2 mV    0.3 Hz   1000 Hz
*/



#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "edflib.h"



#define SMP_FREQ 32000

#define FILE_DURATION 3000


/* if you want to create an EDFplus file instead of BDFplus, outcomment the next line: */
//#define BDF_FORMAT




int main(int argc, char *argv[])
{
  int i, j,
      hdl,
      chns,
      impulse;

  double buf[SMP_FREQ],
         q,
         sine_1,
         sine_305,
         sine_5025,
         sine_8,
         sine_81777,
         sine_85,
         sine_15,
         sine_17,
         sine_1705,
         sine_17025,
         sine_50,
         dc_slope,
         sine_2;



  chns = 23;

#ifdef BDF_FORMAT
  hdl = edfopen_file_writeonly("test_generator11.bdf", EDFLIB_FILETYPE_BDFPLUS, chns);
#else
  hdl = edfopen_file_writeonly("test_generator11.edf", EDFLIB_FILETYPE_EDFPLUS, chns);
#endif

  if(hdl<0)
  {
    printf("error: edfopen_file_writeonly()\n");

    return(1);
  }

//   if(edf_set_datarecord_duration(hdl, 8388))
//   {
//     printf("error: edf_set_datarecord_duration()\n");
//
//     return(1);
//   }

  for(i=0; i<chns; i++)
  {
    if(edf_set_samplefrequency(hdl, i, SMP_FREQ))
    {
      printf("error: edf_set_samplefrequency()\n");

      return(1);
    }
  }

#ifdef BDF_FORMAT
  for(i=0; i<chns; i++)
  {
    if(edf_set_digital_maximum(hdl, i, 8388607))
    {
      printf("error: edf_set_digital_maximum()\n");

      return(1);
    }
  }

  for(i=0; i<chns; i++)
  {
    if(edf_set_digital_minimum(hdl, i, -8388608))
    {
      printf("error: edf_set_digital_minimum()\n");

      return(1);
    }
  }
#else
  for(i=0; i<chns; i++)
  {
    if(edf_set_digital_maximum(hdl, i, 32767))
    {
      printf("error: edf_set_digital_maximum()\n");

      return(1);
    }
  }

  for(i=0; i<chns; i++)
  {
    if(edf_set_digital_minimum(hdl, i, -32768))
    {
      printf("error: edf_set_digital_minimum()\n");

      return(1);
    }
  }
#endif

  for(i=0; i<chns; i++)
  {
    if(edf_set_physical_maximum(hdl, i, 1000.0))
    {
      printf("error: edf_set_physical_maximum()\n");

      return(1);
    }
  }

  if(edf_set_physical_maximum(hdl, 18, -1000.0))
  {
    printf("error: edf_set_physical_maximum()\n");

    return(1);
  }

  for(i=0; i<chns; i++)
  {
    if(edf_set_physical_minimum(hdl, i, -1000.0))
    {
      printf("error: edf_set_physical_minimum()\n");

      return(1);
    }
  }

  if(edf_set_physical_minimum(hdl, 18, 1000.0))
  {
    printf("error: edf_set_physical_minimum()\n");

    return(1);
  }

  for(i=19; i<23; i++)
  {
    if(edf_set_physical_maximum(hdl, i, 6250.0))
    {
      printf("error: edf_set_physical_maximum()\n");

      return(1);
    }

    if(edf_set_physical_minimum(hdl, i, -6250.0))
    {
      printf("error: edf_set_physical_minimum()\n");

      return(1);
    }
  }

  for(i=0; i<chns; i++)
  {
    if(edf_set_physical_dimension(hdl, i, "uV"))
    {
      printf("error: edf_set_physical_dimension()\n");

      return(1);
    }
  }

  if(edf_set_label(hdl, 0, "squarewave"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 1, "ramp"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 2, "pulse"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 3, "noise"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 4, "sine 1 Hz"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 5, "sine 3.05 Hz"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 6, "sine 5.025 Hz"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 7, "sine 8 Hz"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 8, "sine 8.1777 Hz"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 9, "sine 8.5 Hz"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 10, "sine 15 Hz"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 11, "sine 17 Hz"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 12, "sine 17.05 Hz"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 13, "sine 17.025 Hz"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 14, "sine 50 Hz"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 15, "DC +100 uV"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 16, "DC slope"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 17, "sine 2 Hz + DC"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 18, "sine 2 Hz + DC i"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 19, "impulse 3mV"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 20, "trian 20mS 1.5mV"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 21, "tria 200mS 1.5mV"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_label(hdl, 22, "spike 0.5mS 2mV"))
  {
    printf("error: edf_set_label()\n");

    return(1);
  }

  if(edf_set_equipment(hdl, "test generator II"))
  {
    printf("edf_set_equipment()\n");

    return(1);
  }

  sine_1 = 0.0;
  sine_305 = 0.0;
  sine_5025 = 0.0;
  sine_8 = 0.0;
  sine_81777 = 0.0;
  sine_85 = 0.0;
  sine_15 = 0.0;
  sine_17 = 0.0;
  sine_1705 = 0.0;
  sine_17025 = 0.0;
  sine_50 = 0.0;
  dc_slope = 50.0,
  sine_2 = 0.0;
  impulse = 0;

  for(j=0; j<FILE_DURATION; j++)
  {
    if((j%10)<5)                    /* square */
    {
      for(i=0; i<SMP_FREQ; i++)
      {
        buf[i] = 100.0;
      }
    }
    else
    {
      for(i=0; i<SMP_FREQ; i++)
      {
        buf[i] = -100.0;
      }
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

    for(i=0; i<SMP_FREQ; i++)                       /* ramp */
    {
      buf[i] = -100.0 + (i * (200.0 / SMP_FREQ));
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

    for(i=0; i<4; i++)                  /* pulse */
    {
      buf[i] = 100.0;
    }

    for(i=4; i<SMP_FREQ; i++)
    {
      buf[i] = 0.0;
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

    for(i=0; i<SMP_FREQ; i++)                /* noise */
    {
      buf[i] = (int)(100.0 * (rand() / (RAND_MAX + 1.0)));
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

    for(i=0; i<SMP_FREQ; i++)                /* sine 1 Hz */
    {
      q = M_PI * 2.0;
      q /= SMP_FREQ;
      sine_1 += q;
      q = sin(sine_1);
      q *= 100.0;
      buf[i] = q;
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

    for(i=0; i<SMP_FREQ; i++)                /* sine 3.05 Hz */
    {
      q = M_PI * 2.0;
      q /= (SMP_FREQ / 3.05);
      sine_305 += q;
      q = sin(sine_305);
      q *= 100.0;
      buf[i] = q;
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

    for(i=0; i<SMP_FREQ; i++)                /* sine 5.025 Hz */
    {
      q = M_PI * 2.0;
      q /= (SMP_FREQ / 5.025);
      sine_5025 += q;
      q = sin(sine_5025);
      q *= 100.0;
      buf[i] = q;
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

    for(i=0; i<SMP_FREQ; i++)                /* sine 8 Hz */
    {
      q = M_PI * 2.0;
      q /= (SMP_FREQ / 8.0);
      sine_8 += q;
      q = sin(sine_8);
      q *= 100.0;
      buf[i] = q;
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

    for(i=0; i<SMP_FREQ; i++)                /* sine 8.1777 Hz */
    {
      q = M_PI * 2.0;
      q /= (SMP_FREQ / 8.1777);
      sine_81777 += q;
      q = sin(sine_81777);
      q *= 100.0;
      buf[i] = q;
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

    for(i=0; i<SMP_FREQ; i++)                /* sine 8.5 Hz */
    {
      q = M_PI * 2.0;
      q /= (SMP_FREQ / 8.5);
      sine_85 += q;
      q = sin(sine_85);
      q *= 100.0;
      buf[i] = q;
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

    for(i=0; i<SMP_FREQ; i++)                /* sine 15 Hz */
    {
      q = M_PI * 2.0;
      q /= (SMP_FREQ / 15.0);
      sine_15 += q;
      q = sin(sine_15);
      q *= 100.0;
      buf[i] = q;
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

    for(i=0; i<SMP_FREQ; i++)                /* sine 17 Hz */
    {
      q = M_PI * 2.0;
      q /= (SMP_FREQ / 17.0);
      sine_17 += q;
      q = sin(sine_17);
      q *= 100.0;
      buf[i] = q;
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

    for(i=0; i<SMP_FREQ; i++)                /* sine 17.05 Hz */
    {
      q = M_PI * 2.0;
      q /= (SMP_FREQ / 17.05);
      sine_1705 += q;
      q = sin(sine_1705);
      q *= 100.0;
      buf[i] = q;
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

    for(i=0; i<SMP_FREQ; i++)                /* sine 17.025 Hz */
    {
      q = M_PI * 2.0;
      q /= (SMP_FREQ / 17.025);
      sine_17025 += q;
      q = sin(sine_17025);
      q *= 100.0;
      buf[i] = q;
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

    for(i=0; i<SMP_FREQ; i++)                /* sine 50 Hz */
    {
      q = M_PI * 2.0;
      q /= (SMP_FREQ / 50.0);
      sine_50 += q;
      q = sin(sine_50);
      q *= 100.0;
      buf[i] = q;
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

    for(i=0; i<SMP_FREQ; i++)                /* DC */
    {
      buf[i] = 100.0;
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

    for(i=0; i<SMP_FREQ; i++)                /* DC slope */
    {
      buf[i] = dc_slope;

      dc_slope += 0.25;

      if(dc_slope > 1000.0)
      {
        dc_slope = 1000.0;
      }
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

    for(i=0; i<SMP_FREQ; i++)                /* sine 2 Hz + DC */
    {
      q = M_PI * 2.0;
      q /= (SMP_FREQ / 2.0);
      sine_2 += q;
      q = sin(sine_2);
      q *= 1.0;
      buf[i] = q + 500.0;
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

    for(i=0; i<SMP_FREQ; i++)                /* sine 2 Hz + DC (inverted) */
    {
      q = M_PI * 2.0;
      q /= (SMP_FREQ / 2.0);
      sine_2 += q;
      q = sin(sine_2);
      q *= 1.0;
      buf[i] = q + 500.0;
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

    if(impulse == 3)                /* impulse 3 mV 100 mSec. */
    {
      for(i=0; i<(SMP_FREQ / 10); i++)
      {
        buf[i] = 3000.0;
      }

      for(; i<SMP_FREQ; i++)
      {
        buf[i] = 0.0;
      }
    }
    else
    {
      for(i=0; i<SMP_FREQ; i++)
      {
        buf[i] = 0.0;
      }
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

//////////////////////////////////////////////////////////////////////

    if(impulse == 4)                /* triangular 1.5 mV 20 mSec. */
    {
      for(i=0; i<(SMP_FREQ / 100); i++)
      {
        buf[i] = (1500.0 / (SMP_FREQ / 100)) * i;
      }

      for(; i<(SMP_FREQ / 50); i++)
      {
        buf[i] = (1500.0 / (SMP_FREQ / 100)) * ((SMP_FREQ / 100) - i) + 1500.0;
      }

      for(; i<SMP_FREQ; i++)
      {
        buf[i] = 0.0;
      }
    }
    else
    {
      for(i=0; i<SMP_FREQ; i++)
      {
        buf[i] = 0.0;
      }
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

//////////////////////////////////////////////////////////////////////

    if(impulse == 5)                /* triangular 1.5 mV 200 mSec. */
    {
      for(i=0; i<(SMP_FREQ / 10); i++)
      {
        buf[i] = (1500.0 / (SMP_FREQ / 10)) * i;
      }

      for(; i<(SMP_FREQ / 5); i++)
      {
        buf[i] = (1500.0 / (SMP_FREQ / 10)) * ((SMP_FREQ / 10) - i) + 1500.0;
      }

      for(; i<SMP_FREQ; i++)
      {
        buf[i] = 0.0;
      }
    }
    else
    {
      for(i=0; i<SMP_FREQ; i++)
      {
        buf[i] = 0.0;
      }
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

//////////////////////////////////////////////////////////////////////

    for(i=0; i<(SMP_FREQ - (SMP_FREQ / 2000)); i++)  /* spike 2 mV 0.5 mSec. */
    {
      buf[i] = 0.0;
    }

    for(; i<SMP_FREQ; i++)
    {
      buf[i] = 2000.0;
    }

    if(edfwrite_physical_samples(hdl, buf))
    {
      printf("error: edfwrite_physical_samples()\n");

      return(1);
    }

//////////////////////////////////////////////////////////////////////

    impulse++;
    impulse %= 10;
  }

  edfwrite_annotation_latin1(hdl, 0, -1, "Recording starts");

  edfwrite_annotation_latin1(hdl, FILE_DURATION * 10000, -1, "Recording ends");

  edfclose_file(hdl);

  return(0);
}




















