/*
 *
 *  Wireless daemon for Linux
 *
 *  Copyright (C) 2013-2019  Intel Corporation. All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <ell/ell.h>

#include "src/mpdu.h"
#include "src/ie.h"

struct deauthentication_data {
	const unsigned char *frame;
	size_t frame_len;
	uint16_t duration;
	const unsigned char dest[6];
	const unsigned char src[6];
	const unsigned char bssid[6];
	uint8_t fragment_number;
	uint16_t sequence_number;
};

static const unsigned char deauthentication_data_1[] = {
	0xc0, 0x00, 0x3a, 0x01, 0x00, 0x0f, 0xb5, 0x88,
	0xac, 0x82, 0x00, 0x14, 0x6c, 0x7e, 0x40, 0x80,
	0x00, 0x14, 0x6c, 0x7e, 0x40, 0x80, 0x70, 0x22,
	0x01, 0x00
};

static struct deauthentication_data deauthentication_test_1 = {
	.frame = deauthentication_data_1,
	.frame_len = sizeof(deauthentication_data_1),
	.duration = 314,
	.dest = { 0x00, 0x0f, 0xb5, 0x88, 0xac, 0x82 },
	.src = { 0x00, 0x14, 0x6c, 0x7e, 0x40, 0x80 },
	.bssid = { 0x00, 0x14, 0x6c, 0x7e, 0x40, 0x80 },
	.fragment_number = 0,
	.sequence_number = 551,
};

static void deauthentication_test(const void *data)
{
	const struct deauthentication_data *test = data;
	const struct mmpdu_header *mpdu;

	mpdu = mpdu_validate(test->frame, test->frame_len);
	assert(mpdu);

	assert(mpdu->fc.type == MPDU_TYPE_MANAGEMENT);
	assert(mpdu->fc.subtype == MPDU_MANAGEMENT_SUBTYPE_DEAUTHENTICATION);
	assert(mpdu->fc.protocol_version == 0x00);

	assert(L_LE16_TO_CPU(mpdu->duration) == test->duration);
	assert(!memcmp(mpdu->address_1, test->dest, 6));
	assert(!memcmp(mpdu->address_2, test->src, 6));
	assert(!memcmp(mpdu->address_3, test->bssid, 6));

	assert(L_LE16_TO_CPU(mpdu->fragment_number) == test->fragment_number);
	assert(MPDU_SEQUENCE_NUMBER(*mpdu) == test->sequence_number);
}

struct test_frame_data {
	const uint8_t *data;
	size_t len;
	bool good;
};

static const uint8_t probe_req_good1[] = {
	/* Header */
	0x40, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x02, 0x00,
	0x00, 0x00, 0x03, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x50, 0x64,
	/*
	 * SSID, Supported Rates, Extended Supported Rates, DSSS Parameter Set,
	 * HT Capabilities, Vendor Specific, Vendor Specific
	 */
	0x00, 0x05, 0x74, 0x65, 0x73, 0x74, 0x31, 0x01, 0x08, 0x02, 0x04, 0x0b,
	0x16, 0x0c, 0x12, 0x18, 0x24, 0x32, 0x04, 0x30, 0x48, 0x60, 0x6c, 0x03,
	0x01, 0x06, 0x2d, 0x1a, 0x7e, 0x10, 0x1b, 0xff, 0xff, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xdd, 0x69, 0x00, 0x50, 0xf2, 0x04,
	0x10, 0x4a, 0x00, 0x01, 0x10, 0x10, 0x3a, 0x00, 0x01, 0x00, 0x10, 0x08,
	0x00, 0x02, 0x31, 0x48, 0x10, 0x47, 0x00, 0x10, 0xd9, 0xec, 0x65, 0xb2,
	0x32, 0xe4, 0x53, 0x8d, 0xb2, 0x6c, 0x3f, 0x2b, 0x86, 0xf7, 0xa8, 0xd5,
	0x10, 0x54, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x3c, 0x00, 0x01, 0x03, 0x10, 0x02, 0x00, 0x02, 0x00, 0x00, 0x10,
	0x09, 0x00, 0x02, 0x00, 0x00, 0x10, 0x12, 0x00, 0x02, 0x00, 0x00, 0x10,
	0x21, 0x00, 0x01, 0x20, 0x10, 0x23, 0x00, 0x01, 0x20, 0x10, 0x24, 0x00,
	0x01, 0x20, 0x10, 0x11, 0x00, 0x01, 0x20, 0x10, 0x49, 0x00, 0x06, 0x00,
	0x37, 0x2a, 0x00, 0x01, 0x20, 0xdd, 0x11, 0x50, 0x6f, 0x9a, 0x09, 0x02,
	0x02, 0x00, 0x25, 0x00, 0x06, 0x05, 0x00, 0x58, 0x58, 0x04, 0x51, 0x01,
};
static const struct test_frame_data probe_req_good1_data =
	{ probe_req_good1, sizeof(probe_req_good1), true };

static const uint8_t probe_req_good2[] = {
	/* Header */
	0x40, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x02, 0x00,
	0x00, 0x00, 0x03, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x50, 0x64,
	/* SSID, Supported Rates, Extended Supported Rates */
	0x00, 0x05, 0x74, 0x65, 0x73, 0x74, 0x31, 0x01, 0x08, 0x02, 0x04, 0x0b,
	0x16, 0x0c, 0x12, 0x18, 0x24, 0x32, 0x04, 0x30, 0x48, 0x60, 0x6c,
};
static const struct test_frame_data probe_req_good2_data =
	{ probe_req_good2, sizeof(probe_req_good2), true };

static const uint8_t probe_req_ie_out_of_order1[] = {
	/* Header */
	0x40, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x02, 0x00,
	0x00, 0x00, 0x03, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x50, 0x64,
	/* SSID, Extended Supported Rates, Supported Rates */
	0x00, 0x05, 0x74, 0x65, 0x73, 0x74, 0x31, 0x32, 0x04, 0x30, 0x48, 0x60,
	0x6c, 0x01, 0x08, 0x02, 0x04, 0x0b, 0x16, 0x0c, 0x12, 0x18, 0x24,
};

static const struct test_frame_data probe_req_ie_out_of_order1_data = {
	probe_req_ie_out_of_order1, sizeof(probe_req_ie_out_of_order1), true
};

static const uint8_t probe_req_ie_duplicate1[] = {
	/* Header */
	0x40, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x02, 0x00,
	0x00, 0x00, 0x03, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x50, 0x64,
	/* SSID, SSID, Supported Rates, Extended Supported Rates */
	0x00, 0x05, 0x74, 0x65, 0x73, 0x74, 0x31, 0x00, 0x05, 0x74, 0x65, 0x73,
	0x74, 0x31, 0x01, 0x08, 0x02, 0x04, 0x0b, 0x16, 0x0c, 0x12, 0x18, 0x24,
	0x32, 0x04, 0x30, 0x48, 0x60, 0x6c,
};

static const struct test_frame_data probe_req_ie_duplicate1_data = {
	probe_req_ie_duplicate1, sizeof(probe_req_ie_duplicate1), false
};

static const uint8_t probe_req_ie_out_of_order2[] = {
	0x40, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xda, 0xa1,
	0x19, 0x11, 0xec, 0x70, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x50, 0x25,
	0x00, 0x00, 0x01, 0x04, 0x02, 0x04, 0x0b, 0x16, 0x32, 0x08, 0x0c, 0x12,
	0x18, 0x24, 0x30, 0x48, 0x60, 0x6c, 0x03, 0x01, 0x06, 0x2d, 0x1a, 0x6e,
	0x01, 0x03, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0xdd, 0x07, 0x00, 0x50, 0xf2, 0x08, 0x00, 0x01, 0x00, 0x7f, 0x05,
	0x00, 0x00, 0x0a, 0x02, 0x01,
};

static const struct test_frame_data probe_req_ie_out_of_order2_data = {
	probe_req_ie_out_of_order2, sizeof(probe_req_ie_out_of_order2), true
};

static void ie_order_test(const void *data)
{
	const struct test_frame_data *frame = data;

	assert(!!mpdu_validate(frame->data, frame->len) == frame->good);
}

static void ie_sort_test(const void *data)
{
	static uint8_t ie_fils_session[] = { IE_TYPE_EXTENSION, 1, 4 };
	static uint8_t ie_mde[] = { IE_TYPE_MOBILITY_DOMAIN, 0 };
	static uint8_t ie_ssid[] = { IE_TYPE_SSID, 0 };
	static uint8_t ie_rsn[] = { IE_TYPE_RSN, 0 };
	struct iovec ies[64];
	size_t n_ies = 0;
	uint8_t *ie;

	/* Make up some IEs, only the first 3 bytes matter */
	ies[n_ies].iov_base = ie_fils_session;
	ies[n_ies++].iov_len = sizeof(ie_fils_session);

	ies[n_ies].iov_base = ie_mde;
	ies[n_ies++].iov_len = sizeof(ie_mde);

	ies[n_ies].iov_base = ie_ssid;
	ies[n_ies++].iov_len = sizeof(ie_ssid);

	ies[n_ies].iov_base = ie_rsn;
	ies[n_ies++].iov_len = sizeof(ie_rsn);

	mpdu_sort_ies(MPDU_MANAGEMENT_SUBTYPE_ASSOCIATION_REQUEST, ies, n_ies);

	ie = ies[0].iov_base;
	assert(ie[0] == IE_TYPE_SSID);

	ie = ies[1].iov_base;
	assert(ie[0] == IE_TYPE_RSN);

	ie = ies[2].iov_base;
	assert(ie[0] == IE_TYPE_MOBILITY_DOMAIN);

	ie = ies[3].iov_base;
	assert(ie[0] == IE_TYPE_EXTENSION && ie[2] == 4);
}

int main(int argc, char *argv[])
{
	l_test_init(&argc, &argv);

	l_test_add("/Management Frame/Deauthentication Frame 1",
			deauthentication_test, &deauthentication_test_1);

	l_test_add("/IE order/Good 1", ie_order_test, &probe_req_good1_data);
	l_test_add("/IE order/Good 2", ie_order_test, &probe_req_good2_data);
	l_test_add("/IE order/Good (Out of Order IE) 1", ie_order_test,
				&probe_req_ie_out_of_order1_data);
	l_test_add("/IE order/Bad (Duplicate + Out of Order IE) 1",
				ie_order_test,
				&probe_req_ie_duplicate1_data);
	l_test_add("/IE order/Good (Out of Order IE) 2", ie_order_test,
				&probe_req_ie_out_of_order2_data);

	l_test_add("/IE order/Sorting", ie_sort_test, NULL);

	return l_test_run();
}
