/* $Id$ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __CONST_INTEGER_HPP_INCLUDED
#define __CONST_INTEGER_HPP_INCLUDED

#include "frontend/ast/Expression.hpp"
#include "frontend/visitor/Visitor.hpp"
#include "frontend/ast/Types.hpp"
#include "frontend/ast/Name.hpp"

namespace ast {

//! One constant integer.
/** This class represents once constant VHDL integer.
 */
class ConstInteger : public Expression {
public:
	//! c'tor
	/** @param val value of the VHDL integer.
         *  @param unit physical unit.
	 *  @param loc location of the definition.
         */
	ConstInteger(
		universal_integer val,
		Name* unit,
		Location loc);

	//! c'tor
	/** @param val value of the VHDL integer.
         *  @param unit physical unit.
	 *  @param loc location of the definition.
         */
	ConstInteger(
		universal_real val,
		Name* unit,
		Location loc);

	//! c'tor (no physical unit given).
	/** @param val value of the VHDL integer.
	 *  @param loc location of the definition.
         */
	ConstInteger(
		universal_integer val,
		Location loc
		) : 	Expression(BASE_TYPE_INTEGER, loc),
			physUnit(NULL),
			value(val) {}


	//! Accept a Visitor.
 	/** All leaf AST nodes need to implement this method.
         *
         *  @param visitor the Visitor that can visit this node.
         */
	virtual void accept(Visitor& visitor) {
		visitor.visit(*this);
	}

	/** Put a textual representation of the AstNode on the stream.
	 *  @param stream stream to put the textual representation to.
	 */
	virtual void put(std::ostream &stream) const {
		stream << this->value;
	}

	/** Name of the physical unit, if any. */
	Name *physUnit;

	/** Value of the integer. */
	universal_integer value;

private:
	//! determine the value by given Unit Name.
	/** @param unit unit name.
	 *  @param numericValue real/integer factor.
	 *  @return integral value of the constant.
	 */
	template<typename T>
	static universal_integer
	makeValue(Name *unit, T numericValue); 
};

}; /* namespace ast */

#endif /* __CONST_INTEGER_HPP_INCLUDED */
