%% -*- mode: erlang; tab-width: 4; indent-tabs-mode: 1; st-rulers: [70] -*-
%% vim: ts=4 sw=4 ft=erlang noet
%%%-------------------------------------------------------------------
%%% @author Andrew Bennett <potatosaladx@gmail.com>
%%% @copyright 2014-2022, Andrew Bennett
%%% @doc RFC 4648, Section 5: https://tools.ietf.org/html/rfc4648#section-5
%%%
%%% @end
%%% Created :  11 May 2017 by Andrew Bennett <potatosaladx@gmail.com>
%%%-------------------------------------------------------------------
-module(jose_base64url).

-include("jose_base.hrl").

%% API
-export([decode/1]).
-export([decode/2]).
-export(['decode!'/1]).
-export(['decode!'/2]).
-export([encode/1]).
-export([encode/2]).
-export([random/1]).
-export([random/2]).

% Macros
-define(B64URL_TO_INT(C),
	case C of
		$A -> 16#00;
		$B -> 16#01;
		$C -> 16#02;
		$D -> 16#03;
		$E -> 16#04;
		$F -> 16#05;
		$G -> 16#06;
		$H -> 16#07;
		$I -> 16#08;
		$J -> 16#09;
		$K -> 16#0A;
		$L -> 16#0B;
		$M -> 16#0C;
		$N -> 16#0D;
		$O -> 16#0E;
		$P -> 16#0F;
		$Q -> 16#10;
		$R -> 16#11;
		$S -> 16#12;
		$T -> 16#13;
		$U -> 16#14;
		$V -> 16#15;
		$W -> 16#16;
		$X -> 16#17;
		$Y -> 16#18;
		$Z -> 16#19;
		$a -> 16#1A;
		$b -> 16#1B;
		$c -> 16#1C;
		$d -> 16#1D;
		$e -> 16#1E;
		$f -> 16#1F;
		$g -> 16#20;
		$h -> 16#21;
		$i -> 16#22;
		$j -> 16#23;
		$k -> 16#24;
		$l -> 16#25;
		$m -> 16#26;
		$n -> 16#27;
		$o -> 16#28;
		$p -> 16#29;
		$q -> 16#2A;
		$r -> 16#2B;
		$s -> 16#2C;
		$t -> 16#2D;
		$u -> 16#2E;
		$v -> 16#2F;
		$w -> 16#30;
		$x -> 16#31;
		$y -> 16#32;
		$z -> 16#33;
		$0 -> 16#34;
		$1 -> 16#35;
		$2 -> 16#36;
		$3 -> 16#37;
		$4 -> 16#38;
		$5 -> 16#39;
		$6 -> 16#3A;
		$7 -> 16#3B;
		$8 -> 16#3C;
		$9 -> 16#3D;
		$- -> 16#3E;
		$_ -> 16#3F
	end).

%%%===================================================================
%%% API functions
%%%===================================================================

decode(Input) when ?is_iodata(Input) ->
	decode(Input, #{}).

decode(Input, Opts) when ?is_iodata(Input) andalso is_map(Opts) ->
	try 'decode!'(Input, Opts) of
		Output when is_binary(Output) ->
			{ok, Output}
	catch
		_:_ ->
			error
	end;
decode(Input, Opts) when ?is_iodata(Input) andalso is_list(Opts) ->
	decode(Input, maps:from_list(Opts)).

'decode!'(Input) when ?is_iodata(Input) ->
	'decode!'(Input, #{}).

'decode!'([], #{}) ->
	<<>>;
'decode!'(<<>>, #{}) ->
	<<>>;
'decode!'(Input, Opts) when ?is_iodata(Input) andalso is_map(Opts) ->
	Padding = maps:get('padding', Opts, nil),
	Size = erlang:iolist_size(Input),
	Offset =
		case Padding of
			_ when (Padding == false orelse Padding == nil) andalso Size =< 4 ->
				0;
			_ when (Padding == false orelse Padding == nil) andalso (Size rem 4) =/= 0 ->
				Size - (Size rem 4);
			_ when (Padding == false orelse Padding == nil) ->
				Size - 4;
			_ when (Padding == true orelse Padding == nil) andalso Size >= 4 ->
				Size - 4;
			_ ->
				erlang:error({badarg, [Input, Opts]})
		end,
	<< Head0:Offset/binary, Tail0/binary >> = ?to_binary(Input),
	Head = << << (?B64URL_TO_INT(V)):6 >> || << V >> <= Head0 >>,
	Tail =
		case Padding of
			false ->
				case Tail0 of
					<< T0:8, T1:8 >> ->
						<< (?B64URL_TO_INT(T0)):6, (?B64URL_TO_INT(T1) bsr 4):2 >>;
					<< T0:8, T1:8, T2:8 >> ->
						<< (?B64URL_TO_INT(T0)):6, (?B64URL_TO_INT(T1)):6, (?B64URL_TO_INT(T2) bsr 2):4 >>;
					<< T:4/binary >> ->
						<< << (?B64URL_TO_INT(V)):6 >> || << V >> <= T >>;
					<<>> ->
						<<>>;
					_ ->
						erlang:error({badarg, [Input, Opts]})
				end;
			nil ->
				case Tail0 of
					<< T0:8, T1:8, $=, $= >> ->
						<< (?B64URL_TO_INT(T0)):6, (?B64URL_TO_INT(T1) bsr 4):2 >>;
					<< T0:8, T1:8, T2:8, $= >> ->
						<< (?B64URL_TO_INT(T0)):6, (?B64URL_TO_INT(T1)):6, (?B64URL_TO_INT(T2) bsr 2):4 >>;
					<< T0:8, T1:8 >> ->
						<< (?B64URL_TO_INT(T0)):6, (?B64URL_TO_INT(T1) bsr 4):2 >>;
					<< T0:8, T1:8, T2:8 >> ->
						<< (?B64URL_TO_INT(T0)):6, (?B64URL_TO_INT(T1)):6, (?B64URL_TO_INT(T2) bsr 2):4 >>;
					<< T:4/binary >> ->
						<< << (?B64URL_TO_INT(V)):6 >> || << V >> <= T >>;
					<<>> ->
						<<>>
				end;
			true ->
				case Tail0 of
					<< T0:8, T1:8, $=, $= >> ->
						<< (?B64URL_TO_INT(T0)):6, (?B64URL_TO_INT(T1) bsr 4):2 >>;
					<< T0:8, T1:8, T2:8, $= >> ->
						<< (?B64URL_TO_INT(T0)):6, (?B64URL_TO_INT(T1)):6, (?B64URL_TO_INT(T2) bsr 2):4 >>;
					<< T:4/binary >> ->
						<< << (?B64URL_TO_INT(V)):6 >> || << V >> <= T >>;
					<<>> ->
						<<>>;
					_ ->
						erlang:error({badarg, [Input, Opts]})
				end
		end,
	<< Head/binary, Tail/binary >>;
'decode!'(Input, Opts) when ?is_iodata(Input) andalso is_list(Opts) ->
	'decode!'(Input, maps:from_list(Opts)).

encode(Input) when ?is_iodata(Input) ->
	encode(Input, #{}).

encode(Input, Opts) when ?is_iodata(Input) andalso is_map(Opts) ->
	Padding = maps:get('padding', Opts, true),
	Offset = 6 * (erlang:iolist_size(Input) div 6),
	<< Head:Offset/binary, Tail/binary >> = ?to_binary(Input),
	H = << << (encode_pair(V0)):16, (encode_pair(V1)):16, (encode_pair(V2)):16, (encode_pair(V3)):16 >> || << V0:12, V1:12, V2:12, V3:12 >> <= Head >>,
	{T, Pad} =
		case Tail of
			<< T0:12, T1:12, T2:12, T3:4 >> ->
				{<< (encode_pair(T0)):16, (encode_pair(T1)):16, (encode_pair(T2)):16, (encode_char(T3 bsl 2)):8 >>, << $= >>};
			<< T0:12, T1:12, T2:8 >> ->
				{<< (encode_pair(T0)):16, (encode_pair(T1)):16, (encode_pair(T2 bsl 4)):16 >>, << $=, $= >>};
			<< T0:12, T1:12 >> ->
				{<< (encode_pair(T0)):16, (encode_pair(T1)):16 >>, <<>>};
			<< T0:12, T1:4 >> ->
				{<< (encode_pair(T0)):16, (encode_char(T1 bsl 2)):8 >>, <<>>};
			<< T0:8 >> ->
				{<< (encode_pair(T0 bsl 4)):16 >>, << $=, $= >>};
			<<>> ->
				{<<>>, <<>>}
		end,
	case Padding of
		true ->
			<< H/binary, T/binary, Pad/binary >>;
		false ->
			<< H/binary, T/binary >>;
		_ ->
			erlang:error({badarg, [Input, Opts]})
	end;
encode(Input, Opts) when ?is_iodata(Input) andalso is_list(Opts) ->
	encode(Input, maps:from_list(Opts)).

random(Bytes) when is_integer(Bytes) andalso Bytes >= 0 ->
	random(Bytes, #{}).

random(0, Opts) when is_map(Opts) ->
	<<>>;
random(Bytes, Opts) when (Bytes =:= 1) andalso is_map(Opts) ->
	erlang:error({badarg, [Bytes, Opts]});
random(Bytes, Opts) when is_integer(Bytes) andalso Bytes > 0 andalso is_map(Opts) ->
	Padding = maps:get('padding', Opts, true),
	R = (Bytes rem 4),
	Size =
		case Padding of
			true when R =:= 0 ->
				(Bytes * 3) div 4;
			false when R =:= 0 orelse R =:= 2 orelse R =:= 3 ->
				(Bytes * 3) div 4;
			_ ->
				erlang:error({badarg, [Bytes, Opts]})
		end,
	Binary = crypto:strong_rand_bytes(Size),
	encode(Binary, Opts);
random(Bytes, Opts) when is_integer(Bytes) andalso Bytes >= 0 andalso is_list(Opts) ->
	random(Bytes, maps:from_list(Opts)).

%%%-------------------------------------------------------------------
%%% Internal functions
%%%-------------------------------------------------------------------

-compile({inline, [{encode_char, 1}]}).
%% @private
encode_char(V) ->
    element(V + 1, {
        $A,
        $B,
        $C,
        $D,
        $E,
        $F,
        $G,
        $H,
        $I,
        $J,
        $K,
        $L,
        $M,
        $N,
        $O,
        $P,
        $Q,
        $R,
        $S,
        $T,
        $U,
        $V,
        $W,
        $X,
        $Y,
        $Z,
        $a,
        $b,
        $c,
        $d,
        $e,
        $f,
        $g,
        $h,
        $i,
        $j,
        $k,
        $l,
        $m,
        $n,
        $o,
        $p,
        $q,
        $r,
        $s,
        $t,
        $u,
        $v,
        $w,
        $x,
        $y,
        $z,
        $0,
        $1,
        $2,
        $3,
        $4,
        $5,
        $6,
        $7,
        $8,
        $9,
        $-,
        $_
    }).

-compile({inline, [{encode_pair, 1}]}).
%% @private
encode_pair(V) ->
    element(V + 1, {
        $\x{4141},
        $\x{4142},
        $\x{4143},
        $\x{4144},
        $\x{4145},
        $\x{4146},
        $\x{4147},
        $\x{4148},
        $\x{4149},
        $\x{414A},
        $\x{414B},
        $\x{414C},
        $\x{414D},
        $\x{414E},
        $\x{414F},
        $\x{4150},
        $\x{4151},
        $\x{4152},
        $\x{4153},
        $\x{4154},
        $\x{4155},
        $\x{4156},
        $\x{4157},
        $\x{4158},
        $\x{4159},
        $\x{415A},
        $\x{4161},
        $\x{4162},
        $\x{4163},
        $\x{4164},
        $\x{4165},
        $\x{4166},
        $\x{4167},
        $\x{4168},
        $\x{4169},
        $\x{416A},
        $\x{416B},
        $\x{416C},
        $\x{416D},
        $\x{416E},
        $\x{416F},
        $\x{4170},
        $\x{4171},
        $\x{4172},
        $\x{4173},
        $\x{4174},
        $\x{4175},
        $\x{4176},
        $\x{4177},
        $\x{4178},
        $\x{4179},
        $\x{417A},
        $\x{4130},
        $\x{4131},
        $\x{4132},
        $\x{4133},
        $\x{4134},
        $\x{4135},
        $\x{4136},
        $\x{4137},
        $\x{4138},
        $\x{4139},
        $\x{412D},
        $\x{415F},
        $\x{4241},
        $\x{4242},
        $\x{4243},
        $\x{4244},
        $\x{4245},
        $\x{4246},
        $\x{4247},
        $\x{4248},
        $\x{4249},
        $\x{424A},
        $\x{424B},
        $\x{424C},
        $\x{424D},
        $\x{424E},
        $\x{424F},
        $\x{4250},
        $\x{4251},
        $\x{4252},
        $\x{4253},
        $\x{4254},
        $\x{4255},
        $\x{4256},
        $\x{4257},
        $\x{4258},
        $\x{4259},
        $\x{425A},
        $\x{4261},
        $\x{4262},
        $\x{4263},
        $\x{4264},
        $\x{4265},
        $\x{4266},
        $\x{4267},
        $\x{4268},
        $\x{4269},
        $\x{426A},
        $\x{426B},
        $\x{426C},
        $\x{426D},
        $\x{426E},
        $\x{426F},
        $\x{4270},
        $\x{4271},
        $\x{4272},
        $\x{4273},
        $\x{4274},
        $\x{4275},
        $\x{4276},
        $\x{4277},
        $\x{4278},
        $\x{4279},
        $\x{427A},
        $\x{4230},
        $\x{4231},
        $\x{4232},
        $\x{4233},
        $\x{4234},
        $\x{4235},
        $\x{4236},
        $\x{4237},
        $\x{4238},
        $\x{4239},
        $\x{422D},
        $\x{425F},
        $\x{4341},
        $\x{4342},
        $\x{4343},
        $\x{4344},
        $\x{4345},
        $\x{4346},
        $\x{4347},
        $\x{4348},
        $\x{4349},
        $\x{434A},
        $\x{434B},
        $\x{434C},
        $\x{434D},
        $\x{434E},
        $\x{434F},
        $\x{4350},
        $\x{4351},
        $\x{4352},
        $\x{4353},
        $\x{4354},
        $\x{4355},
        $\x{4356},
        $\x{4357},
        $\x{4358},
        $\x{4359},
        $\x{435A},
        $\x{4361},
        $\x{4362},
        $\x{4363},
        $\x{4364},
        $\x{4365},
        $\x{4366},
        $\x{4367},
        $\x{4368},
        $\x{4369},
        $\x{436A},
        $\x{436B},
        $\x{436C},
        $\x{436D},
        $\x{436E},
        $\x{436F},
        $\x{4370},
        $\x{4371},
        $\x{4372},
        $\x{4373},
        $\x{4374},
        $\x{4375},
        $\x{4376},
        $\x{4377},
        $\x{4378},
        $\x{4379},
        $\x{437A},
        $\x{4330},
        $\x{4331},
        $\x{4332},
        $\x{4333},
        $\x{4334},
        $\x{4335},
        $\x{4336},
        $\x{4337},
        $\x{4338},
        $\x{4339},
        $\x{432D},
        $\x{435F},
        $\x{4441},
        $\x{4442},
        $\x{4443},
        $\x{4444},
        $\x{4445},
        $\x{4446},
        $\x{4447},
        $\x{4448},
        $\x{4449},
        $\x{444A},
        $\x{444B},
        $\x{444C},
        $\x{444D},
        $\x{444E},
        $\x{444F},
        $\x{4450},
        $\x{4451},
        $\x{4452},
        $\x{4453},
        $\x{4454},
        $\x{4455},
        $\x{4456},
        $\x{4457},
        $\x{4458},
        $\x{4459},
        $\x{445A},
        $\x{4461},
        $\x{4462},
        $\x{4463},
        $\x{4464},
        $\x{4465},
        $\x{4466},
        $\x{4467},
        $\x{4468},
        $\x{4469},
        $\x{446A},
        $\x{446B},
        $\x{446C},
        $\x{446D},
        $\x{446E},
        $\x{446F},
        $\x{4470},
        $\x{4471},
        $\x{4472},
        $\x{4473},
        $\x{4474},
        $\x{4475},
        $\x{4476},
        $\x{4477},
        $\x{4478},
        $\x{4479},
        $\x{447A},
        $\x{4430},
        $\x{4431},
        $\x{4432},
        $\x{4433},
        $\x{4434},
        $\x{4435},
        $\x{4436},
        $\x{4437},
        $\x{4438},
        $\x{4439},
        $\x{442D},
        $\x{445F},
        $\x{4541},
        $\x{4542},
        $\x{4543},
        $\x{4544},
        $\x{4545},
        $\x{4546},
        $\x{4547},
        $\x{4548},
        $\x{4549},
        $\x{454A},
        $\x{454B},
        $\x{454C},
        $\x{454D},
        $\x{454E},
        $\x{454F},
        $\x{4550},
        $\x{4551},
        $\x{4552},
        $\x{4553},
        $\x{4554},
        $\x{4555},
        $\x{4556},
        $\x{4557},
        $\x{4558},
        $\x{4559},
        $\x{455A},
        $\x{4561},
        $\x{4562},
        $\x{4563},
        $\x{4564},
        $\x{4565},
        $\x{4566},
        $\x{4567},
        $\x{4568},
        $\x{4569},
        $\x{456A},
        $\x{456B},
        $\x{456C},
        $\x{456D},
        $\x{456E},
        $\x{456F},
        $\x{4570},
        $\x{4571},
        $\x{4572},
        $\x{4573},
        $\x{4574},
        $\x{4575},
        $\x{4576},
        $\x{4577},
        $\x{4578},
        $\x{4579},
        $\x{457A},
        $\x{4530},
        $\x{4531},
        $\x{4532},
        $\x{4533},
        $\x{4534},
        $\x{4535},
        $\x{4536},
        $\x{4537},
        $\x{4538},
        $\x{4539},
        $\x{452D},
        $\x{455F},
        $\x{4641},
        $\x{4642},
        $\x{4643},
        $\x{4644},
        $\x{4645},
        $\x{4646},
        $\x{4647},
        $\x{4648},
        $\x{4649},
        $\x{464A},
        $\x{464B},
        $\x{464C},
        $\x{464D},
        $\x{464E},
        $\x{464F},
        $\x{4650},
        $\x{4651},
        $\x{4652},
        $\x{4653},
        $\x{4654},
        $\x{4655},
        $\x{4656},
        $\x{4657},
        $\x{4658},
        $\x{4659},
        $\x{465A},
        $\x{4661},
        $\x{4662},
        $\x{4663},
        $\x{4664},
        $\x{4665},
        $\x{4666},
        $\x{4667},
        $\x{4668},
        $\x{4669},
        $\x{466A},
        $\x{466B},
        $\x{466C},
        $\x{466D},
        $\x{466E},
        $\x{466F},
        $\x{4670},
        $\x{4671},
        $\x{4672},
        $\x{4673},
        $\x{4674},
        $\x{4675},
        $\x{4676},
        $\x{4677},
        $\x{4678},
        $\x{4679},
        $\x{467A},
        $\x{4630},
        $\x{4631},
        $\x{4632},
        $\x{4633},
        $\x{4634},
        $\x{4635},
        $\x{4636},
        $\x{4637},
        $\x{4638},
        $\x{4639},
        $\x{462D},
        $\x{465F},
        $\x{4741},
        $\x{4742},
        $\x{4743},
        $\x{4744},
        $\x{4745},
        $\x{4746},
        $\x{4747},
        $\x{4748},
        $\x{4749},
        $\x{474A},
        $\x{474B},
        $\x{474C},
        $\x{474D},
        $\x{474E},
        $\x{474F},
        $\x{4750},
        $\x{4751},
        $\x{4752},
        $\x{4753},
        $\x{4754},
        $\x{4755},
        $\x{4756},
        $\x{4757},
        $\x{4758},
        $\x{4759},
        $\x{475A},
        $\x{4761},
        $\x{4762},
        $\x{4763},
        $\x{4764},
        $\x{4765},
        $\x{4766},
        $\x{4767},
        $\x{4768},
        $\x{4769},
        $\x{476A},
        $\x{476B},
        $\x{476C},
        $\x{476D},
        $\x{476E},
        $\x{476F},
        $\x{4770},
        $\x{4771},
        $\x{4772},
        $\x{4773},
        $\x{4774},
        $\x{4775},
        $\x{4776},
        $\x{4777},
        $\x{4778},
        $\x{4779},
        $\x{477A},
        $\x{4730},
        $\x{4731},
        $\x{4732},
        $\x{4733},
        $\x{4734},
        $\x{4735},
        $\x{4736},
        $\x{4737},
        $\x{4738},
        $\x{4739},
        $\x{472D},
        $\x{475F},
        $\x{4841},
        $\x{4842},
        $\x{4843},
        $\x{4844},
        $\x{4845},
        $\x{4846},
        $\x{4847},
        $\x{4848},
        $\x{4849},
        $\x{484A},
        $\x{484B},
        $\x{484C},
        $\x{484D},
        $\x{484E},
        $\x{484F},
        $\x{4850},
        $\x{4851},
        $\x{4852},
        $\x{4853},
        $\x{4854},
        $\x{4855},
        $\x{4856},
        $\x{4857},
        $\x{4858},
        $\x{4859},
        $\x{485A},
        $\x{4861},
        $\x{4862},
        $\x{4863},
        $\x{4864},
        $\x{4865},
        $\x{4866},
        $\x{4867},
        $\x{4868},
        $\x{4869},
        $\x{486A},
        $\x{486B},
        $\x{486C},
        $\x{486D},
        $\x{486E},
        $\x{486F},
        $\x{4870},
        $\x{4871},
        $\x{4872},
        $\x{4873},
        $\x{4874},
        $\x{4875},
        $\x{4876},
        $\x{4877},
        $\x{4878},
        $\x{4879},
        $\x{487A},
        $\x{4830},
        $\x{4831},
        $\x{4832},
        $\x{4833},
        $\x{4834},
        $\x{4835},
        $\x{4836},
        $\x{4837},
        $\x{4838},
        $\x{4839},
        $\x{482D},
        $\x{485F},
        $\x{4941},
        $\x{4942},
        $\x{4943},
        $\x{4944},
        $\x{4945},
        $\x{4946},
        $\x{4947},
        $\x{4948},
        $\x{4949},
        $\x{494A},
        $\x{494B},
        $\x{494C},
        $\x{494D},
        $\x{494E},
        $\x{494F},
        $\x{4950},
        $\x{4951},
        $\x{4952},
        $\x{4953},
        $\x{4954},
        $\x{4955},
        $\x{4956},
        $\x{4957},
        $\x{4958},
        $\x{4959},
        $\x{495A},
        $\x{4961},
        $\x{4962},
        $\x{4963},
        $\x{4964},
        $\x{4965},
        $\x{4966},
        $\x{4967},
        $\x{4968},
        $\x{4969},
        $\x{496A},
        $\x{496B},
        $\x{496C},
        $\x{496D},
        $\x{496E},
        $\x{496F},
        $\x{4970},
        $\x{4971},
        $\x{4972},
        $\x{4973},
        $\x{4974},
        $\x{4975},
        $\x{4976},
        $\x{4977},
        $\x{4978},
        $\x{4979},
        $\x{497A},
        $\x{4930},
        $\x{4931},
        $\x{4932},
        $\x{4933},
        $\x{4934},
        $\x{4935},
        $\x{4936},
        $\x{4937},
        $\x{4938},
        $\x{4939},
        $\x{492D},
        $\x{495F},
        $\x{4A41},
        $\x{4A42},
        $\x{4A43},
        $\x{4A44},
        $\x{4A45},
        $\x{4A46},
        $\x{4A47},
        $\x{4A48},
        $\x{4A49},
        $\x{4A4A},
        $\x{4A4B},
        $\x{4A4C},
        $\x{4A4D},
        $\x{4A4E},
        $\x{4A4F},
        $\x{4A50},
        $\x{4A51},
        $\x{4A52},
        $\x{4A53},
        $\x{4A54},
        $\x{4A55},
        $\x{4A56},
        $\x{4A57},
        $\x{4A58},
        $\x{4A59},
        $\x{4A5A},
        $\x{4A61},
        $\x{4A62},
        $\x{4A63},
        $\x{4A64},
        $\x{4A65},
        $\x{4A66},
        $\x{4A67},
        $\x{4A68},
        $\x{4A69},
        $\x{4A6A},
        $\x{4A6B},
        $\x{4A6C},
        $\x{4A6D},
        $\x{4A6E},
        $\x{4A6F},
        $\x{4A70},
        $\x{4A71},
        $\x{4A72},
        $\x{4A73},
        $\x{4A74},
        $\x{4A75},
        $\x{4A76},
        $\x{4A77},
        $\x{4A78},
        $\x{4A79},
        $\x{4A7A},
        $\x{4A30},
        $\x{4A31},
        $\x{4A32},
        $\x{4A33},
        $\x{4A34},
        $\x{4A35},
        $\x{4A36},
        $\x{4A37},
        $\x{4A38},
        $\x{4A39},
        $\x{4A2D},
        $\x{4A5F},
        $\x{4B41},
        $\x{4B42},
        $\x{4B43},
        $\x{4B44},
        $\x{4B45},
        $\x{4B46},
        $\x{4B47},
        $\x{4B48},
        $\x{4B49},
        $\x{4B4A},
        $\x{4B4B},
        $\x{4B4C},
        $\x{4B4D},
        $\x{4B4E},
        $\x{4B4F},
        $\x{4B50},
        $\x{4B51},
        $\x{4B52},
        $\x{4B53},
        $\x{4B54},
        $\x{4B55},
        $\x{4B56},
        $\x{4B57},
        $\x{4B58},
        $\x{4B59},
        $\x{4B5A},
        $\x{4B61},
        $\x{4B62},
        $\x{4B63},
        $\x{4B64},
        $\x{4B65},
        $\x{4B66},
        $\x{4B67},
        $\x{4B68},
        $\x{4B69},
        $\x{4B6A},
        $\x{4B6B},
        $\x{4B6C},
        $\x{4B6D},
        $\x{4B6E},
        $\x{4B6F},
        $\x{4B70},
        $\x{4B71},
        $\x{4B72},
        $\x{4B73},
        $\x{4B74},
        $\x{4B75},
        $\x{4B76},
        $\x{4B77},
        $\x{4B78},
        $\x{4B79},
        $\x{4B7A},
        $\x{4B30},
        $\x{4B31},
        $\x{4B32},
        $\x{4B33},
        $\x{4B34},
        $\x{4B35},
        $\x{4B36},
        $\x{4B37},
        $\x{4B38},
        $\x{4B39},
        $\x{4B2D},
        $\x{4B5F},
        $\x{4C41},
        $\x{4C42},
        $\x{4C43},
        $\x{4C44},
        $\x{4C45},
        $\x{4C46},
        $\x{4C47},
        $\x{4C48},
        $\x{4C49},
        $\x{4C4A},
        $\x{4C4B},
        $\x{4C4C},
        $\x{4C4D},
        $\x{4C4E},
        $\x{4C4F},
        $\x{4C50},
        $\x{4C51},
        $\x{4C52},
        $\x{4C53},
        $\x{4C54},
        $\x{4C55},
        $\x{4C56},
        $\x{4C57},
        $\x{4C58},
        $\x{4C59},
        $\x{4C5A},
        $\x{4C61},
        $\x{4C62},
        $\x{4C63},
        $\x{4C64},
        $\x{4C65},
        $\x{4C66},
        $\x{4C67},
        $\x{4C68},
        $\x{4C69},
        $\x{4C6A},
        $\x{4C6B},
        $\x{4C6C},
        $\x{4C6D},
        $\x{4C6E},
        $\x{4C6F},
        $\x{4C70},
        $\x{4C71},
        $\x{4C72},
        $\x{4C73},
        $\x{4C74},
        $\x{4C75},
        $\x{4C76},
        $\x{4C77},
        $\x{4C78},
        $\x{4C79},
        $\x{4C7A},
        $\x{4C30},
        $\x{4C31},
        $\x{4C32},
        $\x{4C33},
        $\x{4C34},
        $\x{4C35},
        $\x{4C36},
        $\x{4C37},
        $\x{4C38},
        $\x{4C39},
        $\x{4C2D},
        $\x{4C5F},
        $\x{4D41},
        $\x{4D42},
        $\x{4D43},
        $\x{4D44},
        $\x{4D45},
        $\x{4D46},
        $\x{4D47},
        $\x{4D48},
        $\x{4D49},
        $\x{4D4A},
        $\x{4D4B},
        $\x{4D4C},
        $\x{4D4D},
        $\x{4D4E},
        $\x{4D4F},
        $\x{4D50},
        $\x{4D51},
        $\x{4D52},
        $\x{4D53},
        $\x{4D54},
        $\x{4D55},
        $\x{4D56},
        $\x{4D57},
        $\x{4D58},
        $\x{4D59},
        $\x{4D5A},
        $\x{4D61},
        $\x{4D62},
        $\x{4D63},
        $\x{4D64},
        $\x{4D65},
        $\x{4D66},
        $\x{4D67},
        $\x{4D68},
        $\x{4D69},
        $\x{4D6A},
        $\x{4D6B},
        $\x{4D6C},
        $\x{4D6D},
        $\x{4D6E},
        $\x{4D6F},
        $\x{4D70},
        $\x{4D71},
        $\x{4D72},
        $\x{4D73},
        $\x{4D74},
        $\x{4D75},
        $\x{4D76},
        $\x{4D77},
        $\x{4D78},
        $\x{4D79},
        $\x{4D7A},
        $\x{4D30},
        $\x{4D31},
        $\x{4D32},
        $\x{4D33},
        $\x{4D34},
        $\x{4D35},
        $\x{4D36},
        $\x{4D37},
        $\x{4D38},
        $\x{4D39},
        $\x{4D2D},
        $\x{4D5F},
        $\x{4E41},
        $\x{4E42},
        $\x{4E43},
        $\x{4E44},
        $\x{4E45},
        $\x{4E46},
        $\x{4E47},
        $\x{4E48},
        $\x{4E49},
        $\x{4E4A},
        $\x{4E4B},
        $\x{4E4C},
        $\x{4E4D},
        $\x{4E4E},
        $\x{4E4F},
        $\x{4E50},
        $\x{4E51},
        $\x{4E52},
        $\x{4E53},
        $\x{4E54},
        $\x{4E55},
        $\x{4E56},
        $\x{4E57},
        $\x{4E58},
        $\x{4E59},
        $\x{4E5A},
        $\x{4E61},
        $\x{4E62},
        $\x{4E63},
        $\x{4E64},
        $\x{4E65},
        $\x{4E66},
        $\x{4E67},
        $\x{4E68},
        $\x{4E69},
        $\x{4E6A},
        $\x{4E6B},
        $\x{4E6C},
        $\x{4E6D},
        $\x{4E6E},
        $\x{4E6F},
        $\x{4E70},
        $\x{4E71},
        $\x{4E72},
        $\x{4E73},
        $\x{4E74},
        $\x{4E75},
        $\x{4E76},
        $\x{4E77},
        $\x{4E78},
        $\x{4E79},
        $\x{4E7A},
        $\x{4E30},
        $\x{4E31},
        $\x{4E32},
        $\x{4E33},
        $\x{4E34},
        $\x{4E35},
        $\x{4E36},
        $\x{4E37},
        $\x{4E38},
        $\x{4E39},
        $\x{4E2D},
        $\x{4E5F},
        $\x{4F41},
        $\x{4F42},
        $\x{4F43},
        $\x{4F44},
        $\x{4F45},
        $\x{4F46},
        $\x{4F47},
        $\x{4F48},
        $\x{4F49},
        $\x{4F4A},
        $\x{4F4B},
        $\x{4F4C},
        $\x{4F4D},
        $\x{4F4E},
        $\x{4F4F},
        $\x{4F50},
        $\x{4F51},
        $\x{4F52},
        $\x{4F53},
        $\x{4F54},
        $\x{4F55},
        $\x{4F56},
        $\x{4F57},
        $\x{4F58},
        $\x{4F59},
        $\x{4F5A},
        $\x{4F61},
        $\x{4F62},
        $\x{4F63},
        $\x{4F64},
        $\x{4F65},
        $\x{4F66},
        $\x{4F67},
        $\x{4F68},
        $\x{4F69},
        $\x{4F6A},
        $\x{4F6B},
        $\x{4F6C},
        $\x{4F6D},
        $\x{4F6E},
        $\x{4F6F},
        $\x{4F70},
        $\x{4F71},
        $\x{4F72},
        $\x{4F73},
        $\x{4F74},
        $\x{4F75},
        $\x{4F76},
        $\x{4F77},
        $\x{4F78},
        $\x{4F79},
        $\x{4F7A},
        $\x{4F30},
        $\x{4F31},
        $\x{4F32},
        $\x{4F33},
        $\x{4F34},
        $\x{4F35},
        $\x{4F36},
        $\x{4F37},
        $\x{4F38},
        $\x{4F39},
        $\x{4F2D},
        $\x{4F5F},
        $\x{5041},
        $\x{5042},
        $\x{5043},
        $\x{5044},
        $\x{5045},
        $\x{5046},
        $\x{5047},
        $\x{5048},
        $\x{5049},
        $\x{504A},
        $\x{504B},
        $\x{504C},
        $\x{504D},
        $\x{504E},
        $\x{504F},
        $\x{5050},
        $\x{5051},
        $\x{5052},
        $\x{5053},
        $\x{5054},
        $\x{5055},
        $\x{5056},
        $\x{5057},
        $\x{5058},
        $\x{5059},
        $\x{505A},
        $\x{5061},
        $\x{5062},
        $\x{5063},
        $\x{5064},
        $\x{5065},
        $\x{5066},
        $\x{5067},
        $\x{5068},
        $\x{5069},
        $\x{506A},
        $\x{506B},
        $\x{506C},
        $\x{506D},
        $\x{506E},
        $\x{506F},
        $\x{5070},
        $\x{5071},
        $\x{5072},
        $\x{5073},
        $\x{5074},
        $\x{5075},
        $\x{5076},
        $\x{5077},
        $\x{5078},
        $\x{5079},
        $\x{507A},
        $\x{5030},
        $\x{5031},
        $\x{5032},
        $\x{5033},
        $\x{5034},
        $\x{5035},
        $\x{5036},
        $\x{5037},
        $\x{5038},
        $\x{5039},
        $\x{502D},
        $\x{505F},
        $\x{5141},
        $\x{5142},
        $\x{5143},
        $\x{5144},
        $\x{5145},
        $\x{5146},
        $\x{5147},
        $\x{5148},
        $\x{5149},
        $\x{514A},
        $\x{514B},
        $\x{514C},
        $\x{514D},
        $\x{514E},
        $\x{514F},
        $\x{5150},
        $\x{5151},
        $\x{5152},
        $\x{5153},
        $\x{5154},
        $\x{5155},
        $\x{5156},
        $\x{5157},
        $\x{5158},
        $\x{5159},
        $\x{515A},
        $\x{5161},
        $\x{5162},
        $\x{5163},
        $\x{5164},
        $\x{5165},
        $\x{5166},
        $\x{5167},
        $\x{5168},
        $\x{5169},
        $\x{516A},
        $\x{516B},
        $\x{516C},
        $\x{516D},
        $\x{516E},
        $\x{516F},
        $\x{5170},
        $\x{5171},
        $\x{5172},
        $\x{5173},
        $\x{5174},
        $\x{5175},
        $\x{5176},
        $\x{5177},
        $\x{5178},
        $\x{5179},
        $\x{517A},
        $\x{5130},
        $\x{5131},
        $\x{5132},
        $\x{5133},
        $\x{5134},
        $\x{5135},
        $\x{5136},
        $\x{5137},
        $\x{5138},
        $\x{5139},
        $\x{512D},
        $\x{515F},
        $\x{5241},
        $\x{5242},
        $\x{5243},
        $\x{5244},
        $\x{5245},
        $\x{5246},
        $\x{5247},
        $\x{5248},
        $\x{5249},
        $\x{524A},
        $\x{524B},
        $\x{524C},
        $\x{524D},
        $\x{524E},
        $\x{524F},
        $\x{5250},
        $\x{5251},
        $\x{5252},
        $\x{5253},
        $\x{5254},
        $\x{5255},
        $\x{5256},
        $\x{5257},
        $\x{5258},
        $\x{5259},
        $\x{525A},
        $\x{5261},
        $\x{5262},
        $\x{5263},
        $\x{5264},
        $\x{5265},
        $\x{5266},
        $\x{5267},
        $\x{5268},
        $\x{5269},
        $\x{526A},
        $\x{526B},
        $\x{526C},
        $\x{526D},
        $\x{526E},
        $\x{526F},
        $\x{5270},
        $\x{5271},
        $\x{5272},
        $\x{5273},
        $\x{5274},
        $\x{5275},
        $\x{5276},
        $\x{5277},
        $\x{5278},
        $\x{5279},
        $\x{527A},
        $\x{5230},
        $\x{5231},
        $\x{5232},
        $\x{5233},
        $\x{5234},
        $\x{5235},
        $\x{5236},
        $\x{5237},
        $\x{5238},
        $\x{5239},
        $\x{522D},
        $\x{525F},
        $\x{5341},
        $\x{5342},
        $\x{5343},
        $\x{5344},
        $\x{5345},
        $\x{5346},
        $\x{5347},
        $\x{5348},
        $\x{5349},
        $\x{534A},
        $\x{534B},
        $\x{534C},
        $\x{534D},
        $\x{534E},
        $\x{534F},
        $\x{5350},
        $\x{5351},
        $\x{5352},
        $\x{5353},
        $\x{5354},
        $\x{5355},
        $\x{5356},
        $\x{5357},
        $\x{5358},
        $\x{5359},
        $\x{535A},
        $\x{5361},
        $\x{5362},
        $\x{5363},
        $\x{5364},
        $\x{5365},
        $\x{5366},
        $\x{5367},
        $\x{5368},
        $\x{5369},
        $\x{536A},
        $\x{536B},
        $\x{536C},
        $\x{536D},
        $\x{536E},
        $\x{536F},
        $\x{5370},
        $\x{5371},
        $\x{5372},
        $\x{5373},
        $\x{5374},
        $\x{5375},
        $\x{5376},
        $\x{5377},
        $\x{5378},
        $\x{5379},
        $\x{537A},
        $\x{5330},
        $\x{5331},
        $\x{5332},
        $\x{5333},
        $\x{5334},
        $\x{5335},
        $\x{5336},
        $\x{5337},
        $\x{5338},
        $\x{5339},
        $\x{532D},
        $\x{535F},
        $\x{5441},
        $\x{5442},
        $\x{5443},
        $\x{5444},
        $\x{5445},
        $\x{5446},
        $\x{5447},
        $\x{5448},
        $\x{5449},
        $\x{544A},
        $\x{544B},
        $\x{544C},
        $\x{544D},
        $\x{544E},
        $\x{544F},
        $\x{5450},
        $\x{5451},
        $\x{5452},
        $\x{5453},
        $\x{5454},
        $\x{5455},
        $\x{5456},
        $\x{5457},
        $\x{5458},
        $\x{5459},
        $\x{545A},
        $\x{5461},
        $\x{5462},
        $\x{5463},
        $\x{5464},
        $\x{5465},
        $\x{5466},
        $\x{5467},
        $\x{5468},
        $\x{5469},
        $\x{546A},
        $\x{546B},
        $\x{546C},
        $\x{546D},
        $\x{546E},
        $\x{546F},
        $\x{5470},
        $\x{5471},
        $\x{5472},
        $\x{5473},
        $\x{5474},
        $\x{5475},
        $\x{5476},
        $\x{5477},
        $\x{5478},
        $\x{5479},
        $\x{547A},
        $\x{5430},
        $\x{5431},
        $\x{5432},
        $\x{5433},
        $\x{5434},
        $\x{5435},
        $\x{5436},
        $\x{5437},
        $\x{5438},
        $\x{5439},
        $\x{542D},
        $\x{545F},
        $\x{5541},
        $\x{5542},
        $\x{5543},
        $\x{5544},
        $\x{5545},
        $\x{5546},
        $\x{5547},
        $\x{5548},
        $\x{5549},
        $\x{554A},
        $\x{554B},
        $\x{554C},
        $\x{554D},
        $\x{554E},
        $\x{554F},
        $\x{5550},
        $\x{5551},
        $\x{5552},
        $\x{5553},
        $\x{5554},
        $\x{5555},
        $\x{5556},
        $\x{5557},
        $\x{5558},
        $\x{5559},
        $\x{555A},
        $\x{5561},
        $\x{5562},
        $\x{5563},
        $\x{5564},
        $\x{5565},
        $\x{5566},
        $\x{5567},
        $\x{5568},
        $\x{5569},
        $\x{556A},
        $\x{556B},
        $\x{556C},
        $\x{556D},
        $\x{556E},
        $\x{556F},
        $\x{5570},
        $\x{5571},
        $\x{5572},
        $\x{5573},
        $\x{5574},
        $\x{5575},
        $\x{5576},
        $\x{5577},
        $\x{5578},
        $\x{5579},
        $\x{557A},
        $\x{5530},
        $\x{5531},
        $\x{5532},
        $\x{5533},
        $\x{5534},
        $\x{5535},
        $\x{5536},
        $\x{5537},
        $\x{5538},
        $\x{5539},
        $\x{552D},
        $\x{555F},
        $\x{5641},
        $\x{5642},
        $\x{5643},
        $\x{5644},
        $\x{5645},
        $\x{5646},
        $\x{5647},
        $\x{5648},
        $\x{5649},
        $\x{564A},
        $\x{564B},
        $\x{564C},
        $\x{564D},
        $\x{564E},
        $\x{564F},
        $\x{5650},
        $\x{5651},
        $\x{5652},
        $\x{5653},
        $\x{5654},
        $\x{5655},
        $\x{5656},
        $\x{5657},
        $\x{5658},
        $\x{5659},
        $\x{565A},
        $\x{5661},
        $\x{5662},
        $\x{5663},
        $\x{5664},
        $\x{5665},
        $\x{5666},
        $\x{5667},
        $\x{5668},
        $\x{5669},
        $\x{566A},
        $\x{566B},
        $\x{566C},
        $\x{566D},
        $\x{566E},
        $\x{566F},
        $\x{5670},
        $\x{5671},
        $\x{5672},
        $\x{5673},
        $\x{5674},
        $\x{5675},
        $\x{5676},
        $\x{5677},
        $\x{5678},
        $\x{5679},
        $\x{567A},
        $\x{5630},
        $\x{5631},
        $\x{5632},
        $\x{5633},
        $\x{5634},
        $\x{5635},
        $\x{5636},
        $\x{5637},
        $\x{5638},
        $\x{5639},
        $\x{562D},
        $\x{565F},
        $\x{5741},
        $\x{5742},
        $\x{5743},
        $\x{5744},
        $\x{5745},
        $\x{5746},
        $\x{5747},
        $\x{5748},
        $\x{5749},
        $\x{574A},
        $\x{574B},
        $\x{574C},
        $\x{574D},
        $\x{574E},
        $\x{574F},
        $\x{5750},
        $\x{5751},
        $\x{5752},
        $\x{5753},
        $\x{5754},
        $\x{5755},
        $\x{5756},
        $\x{5757},
        $\x{5758},
        $\x{5759},
        $\x{575A},
        $\x{5761},
        $\x{5762},
        $\x{5763},
        $\x{5764},
        $\x{5765},
        $\x{5766},
        $\x{5767},
        $\x{5768},
        $\x{5769},
        $\x{576A},
        $\x{576B},
        $\x{576C},
        $\x{576D},
        $\x{576E},
        $\x{576F},
        $\x{5770},
        $\x{5771},
        $\x{5772},
        $\x{5773},
        $\x{5774},
        $\x{5775},
        $\x{5776},
        $\x{5777},
        $\x{5778},
        $\x{5779},
        $\x{577A},
        $\x{5730},
        $\x{5731},
        $\x{5732},
        $\x{5733},
        $\x{5734},
        $\x{5735},
        $\x{5736},
        $\x{5737},
        $\x{5738},
        $\x{5739},
        $\x{572D},
        $\x{575F},
        $\x{5841},
        $\x{5842},
        $\x{5843},
        $\x{5844},
        $\x{5845},
        $\x{5846},
        $\x{5847},
        $\x{5848},
        $\x{5849},
        $\x{584A},
        $\x{584B},
        $\x{584C},
        $\x{584D},
        $\x{584E},
        $\x{584F},
        $\x{5850},
        $\x{5851},
        $\x{5852},
        $\x{5853},
        $\x{5854},
        $\x{5855},
        $\x{5856},
        $\x{5857},
        $\x{5858},
        $\x{5859},
        $\x{585A},
        $\x{5861},
        $\x{5862},
        $\x{5863},
        $\x{5864},
        $\x{5865},
        $\x{5866},
        $\x{5867},
        $\x{5868},
        $\x{5869},
        $\x{586A},
        $\x{586B},
        $\x{586C},
        $\x{586D},
        $\x{586E},
        $\x{586F},
        $\x{5870},
        $\x{5871},
        $\x{5872},
        $\x{5873},
        $\x{5874},
        $\x{5875},
        $\x{5876},
        $\x{5877},
        $\x{5878},
        $\x{5879},
        $\x{587A},
        $\x{5830},
        $\x{5831},
        $\x{5832},
        $\x{5833},
        $\x{5834},
        $\x{5835},
        $\x{5836},
        $\x{5837},
        $\x{5838},
        $\x{5839},
        $\x{582D},
        $\x{585F},
        $\x{5941},
        $\x{5942},
        $\x{5943},
        $\x{5944},
        $\x{5945},
        $\x{5946},
        $\x{5947},
        $\x{5948},
        $\x{5949},
        $\x{594A},
        $\x{594B},
        $\x{594C},
        $\x{594D},
        $\x{594E},
        $\x{594F},
        $\x{5950},
        $\x{5951},
        $\x{5952},
        $\x{5953},
        $\x{5954},
        $\x{5955},
        $\x{5956},
        $\x{5957},
        $\x{5958},
        $\x{5959},
        $\x{595A},
        $\x{5961},
        $\x{5962},
        $\x{5963},
        $\x{5964},
        $\x{5965},
        $\x{5966},
        $\x{5967},
        $\x{5968},
        $\x{5969},
        $\x{596A},
        $\x{596B},
        $\x{596C},
        $\x{596D},
        $\x{596E},
        $\x{596F},
        $\x{5970},
        $\x{5971},
        $\x{5972},
        $\x{5973},
        $\x{5974},
        $\x{5975},
        $\x{5976},
        $\x{5977},
        $\x{5978},
        $\x{5979},
        $\x{597A},
        $\x{5930},
        $\x{5931},
        $\x{5932},
        $\x{5933},
        $\x{5934},
        $\x{5935},
        $\x{5936},
        $\x{5937},
        $\x{5938},
        $\x{5939},
        $\x{592D},
        $\x{595F},
        $\x{5A41},
        $\x{5A42},
        $\x{5A43},
        $\x{5A44},
        $\x{5A45},
        $\x{5A46},
        $\x{5A47},
        $\x{5A48},
        $\x{5A49},
        $\x{5A4A},
        $\x{5A4B},
        $\x{5A4C},
        $\x{5A4D},
        $\x{5A4E},
        $\x{5A4F},
        $\x{5A50},
        $\x{5A51},
        $\x{5A52},
        $\x{5A53},
        $\x{5A54},
        $\x{5A55},
        $\x{5A56},
        $\x{5A57},
        $\x{5A58},
        $\x{5A59},
        $\x{5A5A},
        $\x{5A61},
        $\x{5A62},
        $\x{5A63},
        $\x{5A64},
        $\x{5A65},
        $\x{5A66},
        $\x{5A67},
        $\x{5A68},
        $\x{5A69},
        $\x{5A6A},
        $\x{5A6B},
        $\x{5A6C},
        $\x{5A6D},
        $\x{5A6E},
        $\x{5A6F},
        $\x{5A70},
        $\x{5A71},
        $\x{5A72},
        $\x{5A73},
        $\x{5A74},
        $\x{5A75},
        $\x{5A76},
        $\x{5A77},
        $\x{5A78},
        $\x{5A79},
        $\x{5A7A},
        $\x{5A30},
        $\x{5A31},
        $\x{5A32},
        $\x{5A33},
        $\x{5A34},
        $\x{5A35},
        $\x{5A36},
        $\x{5A37},
        $\x{5A38},
        $\x{5A39},
        $\x{5A2D},
        $\x{5A5F},
        $\x{6141},
        $\x{6142},
        $\x{6143},
        $\x{6144},
        $\x{6145},
        $\x{6146},
        $\x{6147},
        $\x{6148},
        $\x{6149},
        $\x{614A},
        $\x{614B},
        $\x{614C},
        $\x{614D},
        $\x{614E},
        $\x{614F},
        $\x{6150},
        $\x{6151},
        $\x{6152},
        $\x{6153},
        $\x{6154},
        $\x{6155},
        $\x{6156},
        $\x{6157},
        $\x{6158},
        $\x{6159},
        $\x{615A},
        $\x{6161},
        $\x{6162},
        $\x{6163},
        $\x{6164},
        $\x{6165},
        $\x{6166},
        $\x{6167},
        $\x{6168},
        $\x{6169},
        $\x{616A},
        $\x{616B},
        $\x{616C},
        $\x{616D},
        $\x{616E},
        $\x{616F},
        $\x{6170},
        $\x{6171},
        $\x{6172},
        $\x{6173},
        $\x{6174},
        $\x{6175},
        $\x{6176},
        $\x{6177},
        $\x{6178},
        $\x{6179},
        $\x{617A},
        $\x{6130},
        $\x{6131},
        $\x{6132},
        $\x{6133},
        $\x{6134},
        $\x{6135},
        $\x{6136},
        $\x{6137},
        $\x{6138},
        $\x{6139},
        $\x{612D},
        $\x{615F},
        $\x{6241},
        $\x{6242},
        $\x{6243},
        $\x{6244},
        $\x{6245},
        $\x{6246},
        $\x{6247},
        $\x{6248},
        $\x{6249},
        $\x{624A},
        $\x{624B},
        $\x{624C},
        $\x{624D},
        $\x{624E},
        $\x{624F},
        $\x{6250},
        $\x{6251},
        $\x{6252},
        $\x{6253},
        $\x{6254},
        $\x{6255},
        $\x{6256},
        $\x{6257},
        $\x{6258},
        $\x{6259},
        $\x{625A},
        $\x{6261},
        $\x{6262},
        $\x{6263},
        $\x{6264},
        $\x{6265},
        $\x{6266},
        $\x{6267},
        $\x{6268},
        $\x{6269},
        $\x{626A},
        $\x{626B},
        $\x{626C},
        $\x{626D},
        $\x{626E},
        $\x{626F},
        $\x{6270},
        $\x{6271},
        $\x{6272},
        $\x{6273},
        $\x{6274},
        $\x{6275},
        $\x{6276},
        $\x{6277},
        $\x{6278},
        $\x{6279},
        $\x{627A},
        $\x{6230},
        $\x{6231},
        $\x{6232},
        $\x{6233},
        $\x{6234},
        $\x{6235},
        $\x{6236},
        $\x{6237},
        $\x{6238},
        $\x{6239},
        $\x{622D},
        $\x{625F},
        $\x{6341},
        $\x{6342},
        $\x{6343},
        $\x{6344},
        $\x{6345},
        $\x{6346},
        $\x{6347},
        $\x{6348},
        $\x{6349},
        $\x{634A},
        $\x{634B},
        $\x{634C},
        $\x{634D},
        $\x{634E},
        $\x{634F},
        $\x{6350},
        $\x{6351},
        $\x{6352},
        $\x{6353},
        $\x{6354},
        $\x{6355},
        $\x{6356},
        $\x{6357},
        $\x{6358},
        $\x{6359},
        $\x{635A},
        $\x{6361},
        $\x{6362},
        $\x{6363},
        $\x{6364},
        $\x{6365},
        $\x{6366},
        $\x{6367},
        $\x{6368},
        $\x{6369},
        $\x{636A},
        $\x{636B},
        $\x{636C},
        $\x{636D},
        $\x{636E},
        $\x{636F},
        $\x{6370},
        $\x{6371},
        $\x{6372},
        $\x{6373},
        $\x{6374},
        $\x{6375},
        $\x{6376},
        $\x{6377},
        $\x{6378},
        $\x{6379},
        $\x{637A},
        $\x{6330},
        $\x{6331},
        $\x{6332},
        $\x{6333},
        $\x{6334},
        $\x{6335},
        $\x{6336},
        $\x{6337},
        $\x{6338},
        $\x{6339},
        $\x{632D},
        $\x{635F},
        $\x{6441},
        $\x{6442},
        $\x{6443},
        $\x{6444},
        $\x{6445},
        $\x{6446},
        $\x{6447},
        $\x{6448},
        $\x{6449},
        $\x{644A},
        $\x{644B},
        $\x{644C},
        $\x{644D},
        $\x{644E},
        $\x{644F},
        $\x{6450},
        $\x{6451},
        $\x{6452},
        $\x{6453},
        $\x{6454},
        $\x{6455},
        $\x{6456},
        $\x{6457},
        $\x{6458},
        $\x{6459},
        $\x{645A},
        $\x{6461},
        $\x{6462},
        $\x{6463},
        $\x{6464},
        $\x{6465},
        $\x{6466},
        $\x{6467},
        $\x{6468},
        $\x{6469},
        $\x{646A},
        $\x{646B},
        $\x{646C},
        $\x{646D},
        $\x{646E},
        $\x{646F},
        $\x{6470},
        $\x{6471},
        $\x{6472},
        $\x{6473},
        $\x{6474},
        $\x{6475},
        $\x{6476},
        $\x{6477},
        $\x{6478},
        $\x{6479},
        $\x{647A},
        $\x{6430},
        $\x{6431},
        $\x{6432},
        $\x{6433},
        $\x{6434},
        $\x{6435},
        $\x{6436},
        $\x{6437},
        $\x{6438},
        $\x{6439},
        $\x{642D},
        $\x{645F},
        $\x{6541},
        $\x{6542},
        $\x{6543},
        $\x{6544},
        $\x{6545},
        $\x{6546},
        $\x{6547},
        $\x{6548},
        $\x{6549},
        $\x{654A},
        $\x{654B},
        $\x{654C},
        $\x{654D},
        $\x{654E},
        $\x{654F},
        $\x{6550},
        $\x{6551},
        $\x{6552},
        $\x{6553},
        $\x{6554},
        $\x{6555},
        $\x{6556},
        $\x{6557},
        $\x{6558},
        $\x{6559},
        $\x{655A},
        $\x{6561},
        $\x{6562},
        $\x{6563},
        $\x{6564},
        $\x{6565},
        $\x{6566},
        $\x{6567},
        $\x{6568},
        $\x{6569},
        $\x{656A},
        $\x{656B},
        $\x{656C},
        $\x{656D},
        $\x{656E},
        $\x{656F},
        $\x{6570},
        $\x{6571},
        $\x{6572},
        $\x{6573},
        $\x{6574},
        $\x{6575},
        $\x{6576},
        $\x{6577},
        $\x{6578},
        $\x{6579},
        $\x{657A},
        $\x{6530},
        $\x{6531},
        $\x{6532},
        $\x{6533},
        $\x{6534},
        $\x{6535},
        $\x{6536},
        $\x{6537},
        $\x{6538},
        $\x{6539},
        $\x{652D},
        $\x{655F},
        $\x{6641},
        $\x{6642},
        $\x{6643},
        $\x{6644},
        $\x{6645},
        $\x{6646},
        $\x{6647},
        $\x{6648},
        $\x{6649},
        $\x{664A},
        $\x{664B},
        $\x{664C},
        $\x{664D},
        $\x{664E},
        $\x{664F},
        $\x{6650},
        $\x{6651},
        $\x{6652},
        $\x{6653},
        $\x{6654},
        $\x{6655},
        $\x{6656},
        $\x{6657},
        $\x{6658},
        $\x{6659},
        $\x{665A},
        $\x{6661},
        $\x{6662},
        $\x{6663},
        $\x{6664},
        $\x{6665},
        $\x{6666},
        $\x{6667},
        $\x{6668},
        $\x{6669},
        $\x{666A},
        $\x{666B},
        $\x{666C},
        $\x{666D},
        $\x{666E},
        $\x{666F},
        $\x{6670},
        $\x{6671},
        $\x{6672},
        $\x{6673},
        $\x{6674},
        $\x{6675},
        $\x{6676},
        $\x{6677},
        $\x{6678},
        $\x{6679},
        $\x{667A},
        $\x{6630},
        $\x{6631},
        $\x{6632},
        $\x{6633},
        $\x{6634},
        $\x{6635},
        $\x{6636},
        $\x{6637},
        $\x{6638},
        $\x{6639},
        $\x{662D},
        $\x{665F},
        $\x{6741},
        $\x{6742},
        $\x{6743},
        $\x{6744},
        $\x{6745},
        $\x{6746},
        $\x{6747},
        $\x{6748},
        $\x{6749},
        $\x{674A},
        $\x{674B},
        $\x{674C},
        $\x{674D},
        $\x{674E},
        $\x{674F},
        $\x{6750},
        $\x{6751},
        $\x{6752},
        $\x{6753},
        $\x{6754},
        $\x{6755},
        $\x{6756},
        $\x{6757},
        $\x{6758},
        $\x{6759},
        $\x{675A},
        $\x{6761},
        $\x{6762},
        $\x{6763},
        $\x{6764},
        $\x{6765},
        $\x{6766},
        $\x{6767},
        $\x{6768},
        $\x{6769},
        $\x{676A},
        $\x{676B},
        $\x{676C},
        $\x{676D},
        $\x{676E},
        $\x{676F},
        $\x{6770},
        $\x{6771},
        $\x{6772},
        $\x{6773},
        $\x{6774},
        $\x{6775},
        $\x{6776},
        $\x{6777},
        $\x{6778},
        $\x{6779},
        $\x{677A},
        $\x{6730},
        $\x{6731},
        $\x{6732},
        $\x{6733},
        $\x{6734},
        $\x{6735},
        $\x{6736},
        $\x{6737},
        $\x{6738},
        $\x{6739},
        $\x{672D},
        $\x{675F},
        $\x{6841},
        $\x{6842},
        $\x{6843},
        $\x{6844},
        $\x{6845},
        $\x{6846},
        $\x{6847},
        $\x{6848},
        $\x{6849},
        $\x{684A},
        $\x{684B},
        $\x{684C},
        $\x{684D},
        $\x{684E},
        $\x{684F},
        $\x{6850},
        $\x{6851},
        $\x{6852},
        $\x{6853},
        $\x{6854},
        $\x{6855},
        $\x{6856},
        $\x{6857},
        $\x{6858},
        $\x{6859},
        $\x{685A},
        $\x{6861},
        $\x{6862},
        $\x{6863},
        $\x{6864},
        $\x{6865},
        $\x{6866},
        $\x{6867},
        $\x{6868},
        $\x{6869},
        $\x{686A},
        $\x{686B},
        $\x{686C},
        $\x{686D},
        $\x{686E},
        $\x{686F},
        $\x{6870},
        $\x{6871},
        $\x{6872},
        $\x{6873},
        $\x{6874},
        $\x{6875},
        $\x{6876},
        $\x{6877},
        $\x{6878},
        $\x{6879},
        $\x{687A},
        $\x{6830},
        $\x{6831},
        $\x{6832},
        $\x{6833},
        $\x{6834},
        $\x{6835},
        $\x{6836},
        $\x{6837},
        $\x{6838},
        $\x{6839},
        $\x{682D},
        $\x{685F},
        $\x{6941},
        $\x{6942},
        $\x{6943},
        $\x{6944},
        $\x{6945},
        $\x{6946},
        $\x{6947},
        $\x{6948},
        $\x{6949},
        $\x{694A},
        $\x{694B},
        $\x{694C},
        $\x{694D},
        $\x{694E},
        $\x{694F},
        $\x{6950},
        $\x{6951},
        $\x{6952},
        $\x{6953},
        $\x{6954},
        $\x{6955},
        $\x{6956},
        $\x{6957},
        $\x{6958},
        $\x{6959},
        $\x{695A},
        $\x{6961},
        $\x{6962},
        $\x{6963},
        $\x{6964},
        $\x{6965},
        $\x{6966},
        $\x{6967},
        $\x{6968},
        $\x{6969},
        $\x{696A},
        $\x{696B},
        $\x{696C},
        $\x{696D},
        $\x{696E},
        $\x{696F},
        $\x{6970},
        $\x{6971},
        $\x{6972},
        $\x{6973},
        $\x{6974},
        $\x{6975},
        $\x{6976},
        $\x{6977},
        $\x{6978},
        $\x{6979},
        $\x{697A},
        $\x{6930},
        $\x{6931},
        $\x{6932},
        $\x{6933},
        $\x{6934},
        $\x{6935},
        $\x{6936},
        $\x{6937},
        $\x{6938},
        $\x{6939},
        $\x{692D},
        $\x{695F},
        $\x{6A41},
        $\x{6A42},
        $\x{6A43},
        $\x{6A44},
        $\x{6A45},
        $\x{6A46},
        $\x{6A47},
        $\x{6A48},
        $\x{6A49},
        $\x{6A4A},
        $\x{6A4B},
        $\x{6A4C},
        $\x{6A4D},
        $\x{6A4E},
        $\x{6A4F},
        $\x{6A50},
        $\x{6A51},
        $\x{6A52},
        $\x{6A53},
        $\x{6A54},
        $\x{6A55},
        $\x{6A56},
        $\x{6A57},
        $\x{6A58},
        $\x{6A59},
        $\x{6A5A},
        $\x{6A61},
        $\x{6A62},
        $\x{6A63},
        $\x{6A64},
        $\x{6A65},
        $\x{6A66},
        $\x{6A67},
        $\x{6A68},
        $\x{6A69},
        $\x{6A6A},
        $\x{6A6B},
        $\x{6A6C},
        $\x{6A6D},
        $\x{6A6E},
        $\x{6A6F},
        $\x{6A70},
        $\x{6A71},
        $\x{6A72},
        $\x{6A73},
        $\x{6A74},
        $\x{6A75},
        $\x{6A76},
        $\x{6A77},
        $\x{6A78},
        $\x{6A79},
        $\x{6A7A},
        $\x{6A30},
        $\x{6A31},
        $\x{6A32},
        $\x{6A33},
        $\x{6A34},
        $\x{6A35},
        $\x{6A36},
        $\x{6A37},
        $\x{6A38},
        $\x{6A39},
        $\x{6A2D},
        $\x{6A5F},
        $\x{6B41},
        $\x{6B42},
        $\x{6B43},
        $\x{6B44},
        $\x{6B45},
        $\x{6B46},
        $\x{6B47},
        $\x{6B48},
        $\x{6B49},
        $\x{6B4A},
        $\x{6B4B},
        $\x{6B4C},
        $\x{6B4D},
        $\x{6B4E},
        $\x{6B4F},
        $\x{6B50},
        $\x{6B51},
        $\x{6B52},
        $\x{6B53},
        $\x{6B54},
        $\x{6B55},
        $\x{6B56},
        $\x{6B57},
        $\x{6B58},
        $\x{6B59},
        $\x{6B5A},
        $\x{6B61},
        $\x{6B62},
        $\x{6B63},
        $\x{6B64},
        $\x{6B65},
        $\x{6B66},
        $\x{6B67},
        $\x{6B68},
        $\x{6B69},
        $\x{6B6A},
        $\x{6B6B},
        $\x{6B6C},
        $\x{6B6D},
        $\x{6B6E},
        $\x{6B6F},
        $\x{6B70},
        $\x{6B71},
        $\x{6B72},
        $\x{6B73},
        $\x{6B74},
        $\x{6B75},
        $\x{6B76},
        $\x{6B77},
        $\x{6B78},
        $\x{6B79},
        $\x{6B7A},
        $\x{6B30},
        $\x{6B31},
        $\x{6B32},
        $\x{6B33},
        $\x{6B34},
        $\x{6B35},
        $\x{6B36},
        $\x{6B37},
        $\x{6B38},
        $\x{6B39},
        $\x{6B2D},
        $\x{6B5F},
        $\x{6C41},
        $\x{6C42},
        $\x{6C43},
        $\x{6C44},
        $\x{6C45},
        $\x{6C46},
        $\x{6C47},
        $\x{6C48},
        $\x{6C49},
        $\x{6C4A},
        $\x{6C4B},
        $\x{6C4C},
        $\x{6C4D},
        $\x{6C4E},
        $\x{6C4F},
        $\x{6C50},
        $\x{6C51},
        $\x{6C52},
        $\x{6C53},
        $\x{6C54},
        $\x{6C55},
        $\x{6C56},
        $\x{6C57},
        $\x{6C58},
        $\x{6C59},
        $\x{6C5A},
        $\x{6C61},
        $\x{6C62},
        $\x{6C63},
        $\x{6C64},
        $\x{6C65},
        $\x{6C66},
        $\x{6C67},
        $\x{6C68},
        $\x{6C69},
        $\x{6C6A},
        $\x{6C6B},
        $\x{6C6C},
        $\x{6C6D},
        $\x{6C6E},
        $\x{6C6F},
        $\x{6C70},
        $\x{6C71},
        $\x{6C72},
        $\x{6C73},
        $\x{6C74},
        $\x{6C75},
        $\x{6C76},
        $\x{6C77},
        $\x{6C78},
        $\x{6C79},
        $\x{6C7A},
        $\x{6C30},
        $\x{6C31},
        $\x{6C32},
        $\x{6C33},
        $\x{6C34},
        $\x{6C35},
        $\x{6C36},
        $\x{6C37},
        $\x{6C38},
        $\x{6C39},
        $\x{6C2D},
        $\x{6C5F},
        $\x{6D41},
        $\x{6D42},
        $\x{6D43},
        $\x{6D44},
        $\x{6D45},
        $\x{6D46},
        $\x{6D47},
        $\x{6D48},
        $\x{6D49},
        $\x{6D4A},
        $\x{6D4B},
        $\x{6D4C},
        $\x{6D4D},
        $\x{6D4E},
        $\x{6D4F},
        $\x{6D50},
        $\x{6D51},
        $\x{6D52},
        $\x{6D53},
        $\x{6D54},
        $\x{6D55},
        $\x{6D56},
        $\x{6D57},
        $\x{6D58},
        $\x{6D59},
        $\x{6D5A},
        $\x{6D61},
        $\x{6D62},
        $\x{6D63},
        $\x{6D64},
        $\x{6D65},
        $\x{6D66},
        $\x{6D67},
        $\x{6D68},
        $\x{6D69},
        $\x{6D6A},
        $\x{6D6B},
        $\x{6D6C},
        $\x{6D6D},
        $\x{6D6E},
        $\x{6D6F},
        $\x{6D70},
        $\x{6D71},
        $\x{6D72},
        $\x{6D73},
        $\x{6D74},
        $\x{6D75},
        $\x{6D76},
        $\x{6D77},
        $\x{6D78},
        $\x{6D79},
        $\x{6D7A},
        $\x{6D30},
        $\x{6D31},
        $\x{6D32},
        $\x{6D33},
        $\x{6D34},
        $\x{6D35},
        $\x{6D36},
        $\x{6D37},
        $\x{6D38},
        $\x{6D39},
        $\x{6D2D},
        $\x{6D5F},
        $\x{6E41},
        $\x{6E42},
        $\x{6E43},
        $\x{6E44},
        $\x{6E45},
        $\x{6E46},
        $\x{6E47},
        $\x{6E48},
        $\x{6E49},
        $\x{6E4A},
        $\x{6E4B},
        $\x{6E4C},
        $\x{6E4D},
        $\x{6E4E},
        $\x{6E4F},
        $\x{6E50},
        $\x{6E51},
        $\x{6E52},
        $\x{6E53},
        $\x{6E54},
        $\x{6E55},
        $\x{6E56},
        $\x{6E57},
        $\x{6E58},
        $\x{6E59},
        $\x{6E5A},
        $\x{6E61},
        $\x{6E62},
        $\x{6E63},
        $\x{6E64},
        $\x{6E65},
        $\x{6E66},
        $\x{6E67},
        $\x{6E68},
        $\x{6E69},
        $\x{6E6A},
        $\x{6E6B},
        $\x{6E6C},
        $\x{6E6D},
        $\x{6E6E},
        $\x{6E6F},
        $\x{6E70},
        $\x{6E71},
        $\x{6E72},
        $\x{6E73},
        $\x{6E74},
        $\x{6E75},
        $\x{6E76},
        $\x{6E77},
        $\x{6E78},
        $\x{6E79},
        $\x{6E7A},
        $\x{6E30},
        $\x{6E31},
        $\x{6E32},
        $\x{6E33},
        $\x{6E34},
        $\x{6E35},
        $\x{6E36},
        $\x{6E37},
        $\x{6E38},
        $\x{6E39},
        $\x{6E2D},
        $\x{6E5F},
        $\x{6F41},
        $\x{6F42},
        $\x{6F43},
        $\x{6F44},
        $\x{6F45},
        $\x{6F46},
        $\x{6F47},
        $\x{6F48},
        $\x{6F49},
        $\x{6F4A},
        $\x{6F4B},
        $\x{6F4C},
        $\x{6F4D},
        $\x{6F4E},
        $\x{6F4F},
        $\x{6F50},
        $\x{6F51},
        $\x{6F52},
        $\x{6F53},
        $\x{6F54},
        $\x{6F55},
        $\x{6F56},
        $\x{6F57},
        $\x{6F58},
        $\x{6F59},
        $\x{6F5A},
        $\x{6F61},
        $\x{6F62},
        $\x{6F63},
        $\x{6F64},
        $\x{6F65},
        $\x{6F66},
        $\x{6F67},
        $\x{6F68},
        $\x{6F69},
        $\x{6F6A},
        $\x{6F6B},
        $\x{6F6C},
        $\x{6F6D},
        $\x{6F6E},
        $\x{6F6F},
        $\x{6F70},
        $\x{6F71},
        $\x{6F72},
        $\x{6F73},
        $\x{6F74},
        $\x{6F75},
        $\x{6F76},
        $\x{6F77},
        $\x{6F78},
        $\x{6F79},
        $\x{6F7A},
        $\x{6F30},
        $\x{6F31},
        $\x{6F32},
        $\x{6F33},
        $\x{6F34},
        $\x{6F35},
        $\x{6F36},
        $\x{6F37},
        $\x{6F38},
        $\x{6F39},
        $\x{6F2D},
        $\x{6F5F},
        $\x{7041},
        $\x{7042},
        $\x{7043},
        $\x{7044},
        $\x{7045},
        $\x{7046},
        $\x{7047},
        $\x{7048},
        $\x{7049},
        $\x{704A},
        $\x{704B},
        $\x{704C},
        $\x{704D},
        $\x{704E},
        $\x{704F},
        $\x{7050},
        $\x{7051},
        $\x{7052},
        $\x{7053},
        $\x{7054},
        $\x{7055},
        $\x{7056},
        $\x{7057},
        $\x{7058},
        $\x{7059},
        $\x{705A},
        $\x{7061},
        $\x{7062},
        $\x{7063},
        $\x{7064},
        $\x{7065},
        $\x{7066},
        $\x{7067},
        $\x{7068},
        $\x{7069},
        $\x{706A},
        $\x{706B},
        $\x{706C},
        $\x{706D},
        $\x{706E},
        $\x{706F},
        $\x{7070},
        $\x{7071},
        $\x{7072},
        $\x{7073},
        $\x{7074},
        $\x{7075},
        $\x{7076},
        $\x{7077},
        $\x{7078},
        $\x{7079},
        $\x{707A},
        $\x{7030},
        $\x{7031},
        $\x{7032},
        $\x{7033},
        $\x{7034},
        $\x{7035},
        $\x{7036},
        $\x{7037},
        $\x{7038},
        $\x{7039},
        $\x{702D},
        $\x{705F},
        $\x{7141},
        $\x{7142},
        $\x{7143},
        $\x{7144},
        $\x{7145},
        $\x{7146},
        $\x{7147},
        $\x{7148},
        $\x{7149},
        $\x{714A},
        $\x{714B},
        $\x{714C},
        $\x{714D},
        $\x{714E},
        $\x{714F},
        $\x{7150},
        $\x{7151},
        $\x{7152},
        $\x{7153},
        $\x{7154},
        $\x{7155},
        $\x{7156},
        $\x{7157},
        $\x{7158},
        $\x{7159},
        $\x{715A},
        $\x{7161},
        $\x{7162},
        $\x{7163},
        $\x{7164},
        $\x{7165},
        $\x{7166},
        $\x{7167},
        $\x{7168},
        $\x{7169},
        $\x{716A},
        $\x{716B},
        $\x{716C},
        $\x{716D},
        $\x{716E},
        $\x{716F},
        $\x{7170},
        $\x{7171},
        $\x{7172},
        $\x{7173},
        $\x{7174},
        $\x{7175},
        $\x{7176},
        $\x{7177},
        $\x{7178},
        $\x{7179},
        $\x{717A},
        $\x{7130},
        $\x{7131},
        $\x{7132},
        $\x{7133},
        $\x{7134},
        $\x{7135},
        $\x{7136},
        $\x{7137},
        $\x{7138},
        $\x{7139},
        $\x{712D},
        $\x{715F},
        $\x{7241},
        $\x{7242},
        $\x{7243},
        $\x{7244},
        $\x{7245},
        $\x{7246},
        $\x{7247},
        $\x{7248},
        $\x{7249},
        $\x{724A},
        $\x{724B},
        $\x{724C},
        $\x{724D},
        $\x{724E},
        $\x{724F},
        $\x{7250},
        $\x{7251},
        $\x{7252},
        $\x{7253},
        $\x{7254},
        $\x{7255},
        $\x{7256},
        $\x{7257},
        $\x{7258},
        $\x{7259},
        $\x{725A},
        $\x{7261},
        $\x{7262},
        $\x{7263},
        $\x{7264},
        $\x{7265},
        $\x{7266},
        $\x{7267},
        $\x{7268},
        $\x{7269},
        $\x{726A},
        $\x{726B},
        $\x{726C},
        $\x{726D},
        $\x{726E},
        $\x{726F},
        $\x{7270},
        $\x{7271},
        $\x{7272},
        $\x{7273},
        $\x{7274},
        $\x{7275},
        $\x{7276},
        $\x{7277},
        $\x{7278},
        $\x{7279},
        $\x{727A},
        $\x{7230},
        $\x{7231},
        $\x{7232},
        $\x{7233},
        $\x{7234},
        $\x{7235},
        $\x{7236},
        $\x{7237},
        $\x{7238},
        $\x{7239},
        $\x{722D},
        $\x{725F},
        $\x{7341},
        $\x{7342},
        $\x{7343},
        $\x{7344},
        $\x{7345},
        $\x{7346},
        $\x{7347},
        $\x{7348},
        $\x{7349},
        $\x{734A},
        $\x{734B},
        $\x{734C},
        $\x{734D},
        $\x{734E},
        $\x{734F},
        $\x{7350},
        $\x{7351},
        $\x{7352},
        $\x{7353},
        $\x{7354},
        $\x{7355},
        $\x{7356},
        $\x{7357},
        $\x{7358},
        $\x{7359},
        $\x{735A},
        $\x{7361},
        $\x{7362},
        $\x{7363},
        $\x{7364},
        $\x{7365},
        $\x{7366},
        $\x{7367},
        $\x{7368},
        $\x{7369},
        $\x{736A},
        $\x{736B},
        $\x{736C},
        $\x{736D},
        $\x{736E},
        $\x{736F},
        $\x{7370},
        $\x{7371},
        $\x{7372},
        $\x{7373},
        $\x{7374},
        $\x{7375},
        $\x{7376},
        $\x{7377},
        $\x{7378},
        $\x{7379},
        $\x{737A},
        $\x{7330},
        $\x{7331},
        $\x{7332},
        $\x{7333},
        $\x{7334},
        $\x{7335},
        $\x{7336},
        $\x{7337},
        $\x{7338},
        $\x{7339},
        $\x{732D},
        $\x{735F},
        $\x{7441},
        $\x{7442},
        $\x{7443},
        $\x{7444},
        $\x{7445},
        $\x{7446},
        $\x{7447},
        $\x{7448},
        $\x{7449},
        $\x{744A},
        $\x{744B},
        $\x{744C},
        $\x{744D},
        $\x{744E},
        $\x{744F},
        $\x{7450},
        $\x{7451},
        $\x{7452},
        $\x{7453},
        $\x{7454},
        $\x{7455},
        $\x{7456},
        $\x{7457},
        $\x{7458},
        $\x{7459},
        $\x{745A},
        $\x{7461},
        $\x{7462},
        $\x{7463},
        $\x{7464},
        $\x{7465},
        $\x{7466},
        $\x{7467},
        $\x{7468},
        $\x{7469},
        $\x{746A},
        $\x{746B},
        $\x{746C},
        $\x{746D},
        $\x{746E},
        $\x{746F},
        $\x{7470},
        $\x{7471},
        $\x{7472},
        $\x{7473},
        $\x{7474},
        $\x{7475},
        $\x{7476},
        $\x{7477},
        $\x{7478},
        $\x{7479},
        $\x{747A},
        $\x{7430},
        $\x{7431},
        $\x{7432},
        $\x{7433},
        $\x{7434},
        $\x{7435},
        $\x{7436},
        $\x{7437},
        $\x{7438},
        $\x{7439},
        $\x{742D},
        $\x{745F},
        $\x{7541},
        $\x{7542},
        $\x{7543},
        $\x{7544},
        $\x{7545},
        $\x{7546},
        $\x{7547},
        $\x{7548},
        $\x{7549},
        $\x{754A},
        $\x{754B},
        $\x{754C},
        $\x{754D},
        $\x{754E},
        $\x{754F},
        $\x{7550},
        $\x{7551},
        $\x{7552},
        $\x{7553},
        $\x{7554},
        $\x{7555},
        $\x{7556},
        $\x{7557},
        $\x{7558},
        $\x{7559},
        $\x{755A},
        $\x{7561},
        $\x{7562},
        $\x{7563},
        $\x{7564},
        $\x{7565},
        $\x{7566},
        $\x{7567},
        $\x{7568},
        $\x{7569},
        $\x{756A},
        $\x{756B},
        $\x{756C},
        $\x{756D},
        $\x{756E},
        $\x{756F},
        $\x{7570},
        $\x{7571},
        $\x{7572},
        $\x{7573},
        $\x{7574},
        $\x{7575},
        $\x{7576},
        $\x{7577},
        $\x{7578},
        $\x{7579},
        $\x{757A},
        $\x{7530},
        $\x{7531},
        $\x{7532},
        $\x{7533},
        $\x{7534},
        $\x{7535},
        $\x{7536},
        $\x{7537},
        $\x{7538},
        $\x{7539},
        $\x{752D},
        $\x{755F},
        $\x{7641},
        $\x{7642},
        $\x{7643},
        $\x{7644},
        $\x{7645},
        $\x{7646},
        $\x{7647},
        $\x{7648},
        $\x{7649},
        $\x{764A},
        $\x{764B},
        $\x{764C},
        $\x{764D},
        $\x{764E},
        $\x{764F},
        $\x{7650},
        $\x{7651},
        $\x{7652},
        $\x{7653},
        $\x{7654},
        $\x{7655},
        $\x{7656},
        $\x{7657},
        $\x{7658},
        $\x{7659},
        $\x{765A},
        $\x{7661},
        $\x{7662},
        $\x{7663},
        $\x{7664},
        $\x{7665},
        $\x{7666},
        $\x{7667},
        $\x{7668},
        $\x{7669},
        $\x{766A},
        $\x{766B},
        $\x{766C},
        $\x{766D},
        $\x{766E},
        $\x{766F},
        $\x{7670},
        $\x{7671},
        $\x{7672},
        $\x{7673},
        $\x{7674},
        $\x{7675},
        $\x{7676},
        $\x{7677},
        $\x{7678},
        $\x{7679},
        $\x{767A},
        $\x{7630},
        $\x{7631},
        $\x{7632},
        $\x{7633},
        $\x{7634},
        $\x{7635},
        $\x{7636},
        $\x{7637},
        $\x{7638},
        $\x{7639},
        $\x{762D},
        $\x{765F},
        $\x{7741},
        $\x{7742},
        $\x{7743},
        $\x{7744},
        $\x{7745},
        $\x{7746},
        $\x{7747},
        $\x{7748},
        $\x{7749},
        $\x{774A},
        $\x{774B},
        $\x{774C},
        $\x{774D},
        $\x{774E},
        $\x{774F},
        $\x{7750},
        $\x{7751},
        $\x{7752},
        $\x{7753},
        $\x{7754},
        $\x{7755},
        $\x{7756},
        $\x{7757},
        $\x{7758},
        $\x{7759},
        $\x{775A},
        $\x{7761},
        $\x{7762},
        $\x{7763},
        $\x{7764},
        $\x{7765},
        $\x{7766},
        $\x{7767},
        $\x{7768},
        $\x{7769},
        $\x{776A},
        $\x{776B},
        $\x{776C},
        $\x{776D},
        $\x{776E},
        $\x{776F},
        $\x{7770},
        $\x{7771},
        $\x{7772},
        $\x{7773},
        $\x{7774},
        $\x{7775},
        $\x{7776},
        $\x{7777},
        $\x{7778},
        $\x{7779},
        $\x{777A},
        $\x{7730},
        $\x{7731},
        $\x{7732},
        $\x{7733},
        $\x{7734},
        $\x{7735},
        $\x{7736},
        $\x{7737},
        $\x{7738},
        $\x{7739},
        $\x{772D},
        $\x{775F},
        $\x{7841},
        $\x{7842},
        $\x{7843},
        $\x{7844},
        $\x{7845},
        $\x{7846},
        $\x{7847},
        $\x{7848},
        $\x{7849},
        $\x{784A},
        $\x{784B},
        $\x{784C},
        $\x{784D},
        $\x{784E},
        $\x{784F},
        $\x{7850},
        $\x{7851},
        $\x{7852},
        $\x{7853},
        $\x{7854},
        $\x{7855},
        $\x{7856},
        $\x{7857},
        $\x{7858},
        $\x{7859},
        $\x{785A},
        $\x{7861},
        $\x{7862},
        $\x{7863},
        $\x{7864},
        $\x{7865},
        $\x{7866},
        $\x{7867},
        $\x{7868},
        $\x{7869},
        $\x{786A},
        $\x{786B},
        $\x{786C},
        $\x{786D},
        $\x{786E},
        $\x{786F},
        $\x{7870},
        $\x{7871},
        $\x{7872},
        $\x{7873},
        $\x{7874},
        $\x{7875},
        $\x{7876},
        $\x{7877},
        $\x{7878},
        $\x{7879},
        $\x{787A},
        $\x{7830},
        $\x{7831},
        $\x{7832},
        $\x{7833},
        $\x{7834},
        $\x{7835},
        $\x{7836},
        $\x{7837},
        $\x{7838},
        $\x{7839},
        $\x{782D},
        $\x{785F},
        $\x{7941},
        $\x{7942},
        $\x{7943},
        $\x{7944},
        $\x{7945},
        $\x{7946},
        $\x{7947},
        $\x{7948},
        $\x{7949},
        $\x{794A},
        $\x{794B},
        $\x{794C},
        $\x{794D},
        $\x{794E},
        $\x{794F},
        $\x{7950},
        $\x{7951},
        $\x{7952},
        $\x{7953},
        $\x{7954},
        $\x{7955},
        $\x{7956},
        $\x{7957},
        $\x{7958},
        $\x{7959},
        $\x{795A},
        $\x{7961},
        $\x{7962},
        $\x{7963},
        $\x{7964},
        $\x{7965},
        $\x{7966},
        $\x{7967},
        $\x{7968},
        $\x{7969},
        $\x{796A},
        $\x{796B},
        $\x{796C},
        $\x{796D},
        $\x{796E},
        $\x{796F},
        $\x{7970},
        $\x{7971},
        $\x{7972},
        $\x{7973},
        $\x{7974},
        $\x{7975},
        $\x{7976},
        $\x{7977},
        $\x{7978},
        $\x{7979},
        $\x{797A},
        $\x{7930},
        $\x{7931},
        $\x{7932},
        $\x{7933},
        $\x{7934},
        $\x{7935},
        $\x{7936},
        $\x{7937},
        $\x{7938},
        $\x{7939},
        $\x{792D},
        $\x{795F},
        $\x{7A41},
        $\x{7A42},
        $\x{7A43},
        $\x{7A44},
        $\x{7A45},
        $\x{7A46},
        $\x{7A47},
        $\x{7A48},
        $\x{7A49},
        $\x{7A4A},
        $\x{7A4B},
        $\x{7A4C},
        $\x{7A4D},
        $\x{7A4E},
        $\x{7A4F},
        $\x{7A50},
        $\x{7A51},
        $\x{7A52},
        $\x{7A53},
        $\x{7A54},
        $\x{7A55},
        $\x{7A56},
        $\x{7A57},
        $\x{7A58},
        $\x{7A59},
        $\x{7A5A},
        $\x{7A61},
        $\x{7A62},
        $\x{7A63},
        $\x{7A64},
        $\x{7A65},
        $\x{7A66},
        $\x{7A67},
        $\x{7A68},
        $\x{7A69},
        $\x{7A6A},
        $\x{7A6B},
        $\x{7A6C},
        $\x{7A6D},
        $\x{7A6E},
        $\x{7A6F},
        $\x{7A70},
        $\x{7A71},
        $\x{7A72},
        $\x{7A73},
        $\x{7A74},
        $\x{7A75},
        $\x{7A76},
        $\x{7A77},
        $\x{7A78},
        $\x{7A79},
        $\x{7A7A},
        $\x{7A30},
        $\x{7A31},
        $\x{7A32},
        $\x{7A33},
        $\x{7A34},
        $\x{7A35},
        $\x{7A36},
        $\x{7A37},
        $\x{7A38},
        $\x{7A39},
        $\x{7A2D},
        $\x{7A5F},
        $\x{3041},
        $\x{3042},
        $\x{3043},
        $\x{3044},
        $\x{3045},
        $\x{3046},
        $\x{3047},
        $\x{3048},
        $\x{3049},
        $\x{304A},
        $\x{304B},
        $\x{304C},
        $\x{304D},
        $\x{304E},
        $\x{304F},
        $\x{3050},
        $\x{3051},
        $\x{3052},
        $\x{3053},
        $\x{3054},
        $\x{3055},
        $\x{3056},
        $\x{3057},
        $\x{3058},
        $\x{3059},
        $\x{305A},
        $\x{3061},
        $\x{3062},
        $\x{3063},
        $\x{3064},
        $\x{3065},
        $\x{3066},
        $\x{3067},
        $\x{3068},
        $\x{3069},
        $\x{306A},
        $\x{306B},
        $\x{306C},
        $\x{306D},
        $\x{306E},
        $\x{306F},
        $\x{3070},
        $\x{3071},
        $\x{3072},
        $\x{3073},
        $\x{3074},
        $\x{3075},
        $\x{3076},
        $\x{3077},
        $\x{3078},
        $\x{3079},
        $\x{307A},
        $\x{3030},
        $\x{3031},
        $\x{3032},
        $\x{3033},
        $\x{3034},
        $\x{3035},
        $\x{3036},
        $\x{3037},
        $\x{3038},
        $\x{3039},
        $\x{302D},
        $\x{305F},
        $\x{3141},
        $\x{3142},
        $\x{3143},
        $\x{3144},
        $\x{3145},
        $\x{3146},
        $\x{3147},
        $\x{3148},
        $\x{3149},
        $\x{314A},
        $\x{314B},
        $\x{314C},
        $\x{314D},
        $\x{314E},
        $\x{314F},
        $\x{3150},
        $\x{3151},
        $\x{3152},
        $\x{3153},
        $\x{3154},
        $\x{3155},
        $\x{3156},
        $\x{3157},
        $\x{3158},
        $\x{3159},
        $\x{315A},
        $\x{3161},
        $\x{3162},
        $\x{3163},
        $\x{3164},
        $\x{3165},
        $\x{3166},
        $\x{3167},
        $\x{3168},
        $\x{3169},
        $\x{316A},
        $\x{316B},
        $\x{316C},
        $\x{316D},
        $\x{316E},
        $\x{316F},
        $\x{3170},
        $\x{3171},
        $\x{3172},
        $\x{3173},
        $\x{3174},
        $\x{3175},
        $\x{3176},
        $\x{3177},
        $\x{3178},
        $\x{3179},
        $\x{317A},
        $\x{3130},
        $\x{3131},
        $\x{3132},
        $\x{3133},
        $\x{3134},
        $\x{3135},
        $\x{3136},
        $\x{3137},
        $\x{3138},
        $\x{3139},
        $\x{312D},
        $\x{315F},
        $\x{3241},
        $\x{3242},
        $\x{3243},
        $\x{3244},
        $\x{3245},
        $\x{3246},
        $\x{3247},
        $\x{3248},
        $\x{3249},
        $\x{324A},
        $\x{324B},
        $\x{324C},
        $\x{324D},
        $\x{324E},
        $\x{324F},
        $\x{3250},
        $\x{3251},
        $\x{3252},
        $\x{3253},
        $\x{3254},
        $\x{3255},
        $\x{3256},
        $\x{3257},
        $\x{3258},
        $\x{3259},
        $\x{325A},
        $\x{3261},
        $\x{3262},
        $\x{3263},
        $\x{3264},
        $\x{3265},
        $\x{3266},
        $\x{3267},
        $\x{3268},
        $\x{3269},
        $\x{326A},
        $\x{326B},
        $\x{326C},
        $\x{326D},
        $\x{326E},
        $\x{326F},
        $\x{3270},
        $\x{3271},
        $\x{3272},
        $\x{3273},
        $\x{3274},
        $\x{3275},
        $\x{3276},
        $\x{3277},
        $\x{3278},
        $\x{3279},
        $\x{327A},
        $\x{3230},
        $\x{3231},
        $\x{3232},
        $\x{3233},
        $\x{3234},
        $\x{3235},
        $\x{3236},
        $\x{3237},
        $\x{3238},
        $\x{3239},
        $\x{322D},
        $\x{325F},
        $\x{3341},
        $\x{3342},
        $\x{3343},
        $\x{3344},
        $\x{3345},
        $\x{3346},
        $\x{3347},
        $\x{3348},
        $\x{3349},
        $\x{334A},
        $\x{334B},
        $\x{334C},
        $\x{334D},
        $\x{334E},
        $\x{334F},
        $\x{3350},
        $\x{3351},
        $\x{3352},
        $\x{3353},
        $\x{3354},
        $\x{3355},
        $\x{3356},
        $\x{3357},
        $\x{3358},
        $\x{3359},
        $\x{335A},
        $\x{3361},
        $\x{3362},
        $\x{3363},
        $\x{3364},
        $\x{3365},
        $\x{3366},
        $\x{3367},
        $\x{3368},
        $\x{3369},
        $\x{336A},
        $\x{336B},
        $\x{336C},
        $\x{336D},
        $\x{336E},
        $\x{336F},
        $\x{3370},
        $\x{3371},
        $\x{3372},
        $\x{3373},
        $\x{3374},
        $\x{3375},
        $\x{3376},
        $\x{3377},
        $\x{3378},
        $\x{3379},
        $\x{337A},
        $\x{3330},
        $\x{3331},
        $\x{3332},
        $\x{3333},
        $\x{3334},
        $\x{3335},
        $\x{3336},
        $\x{3337},
        $\x{3338},
        $\x{3339},
        $\x{332D},
        $\x{335F},
        $\x{3441},
        $\x{3442},
        $\x{3443},
        $\x{3444},
        $\x{3445},
        $\x{3446},
        $\x{3447},
        $\x{3448},
        $\x{3449},
        $\x{344A},
        $\x{344B},
        $\x{344C},
        $\x{344D},
        $\x{344E},
        $\x{344F},
        $\x{3450},
        $\x{3451},
        $\x{3452},
        $\x{3453},
        $\x{3454},
        $\x{3455},
        $\x{3456},
        $\x{3457},
        $\x{3458},
        $\x{3459},
        $\x{345A},
        $\x{3461},
        $\x{3462},
        $\x{3463},
        $\x{3464},
        $\x{3465},
        $\x{3466},
        $\x{3467},
        $\x{3468},
        $\x{3469},
        $\x{346A},
        $\x{346B},
        $\x{346C},
        $\x{346D},
        $\x{346E},
        $\x{346F},
        $\x{3470},
        $\x{3471},
        $\x{3472},
        $\x{3473},
        $\x{3474},
        $\x{3475},
        $\x{3476},
        $\x{3477},
        $\x{3478},
        $\x{3479},
        $\x{347A},
        $\x{3430},
        $\x{3431},
        $\x{3432},
        $\x{3433},
        $\x{3434},
        $\x{3435},
        $\x{3436},
        $\x{3437},
        $\x{3438},
        $\x{3439},
        $\x{342D},
        $\x{345F},
        $\x{3541},
        $\x{3542},
        $\x{3543},
        $\x{3544},
        $\x{3545},
        $\x{3546},
        $\x{3547},
        $\x{3548},
        $\x{3549},
        $\x{354A},
        $\x{354B},
        $\x{354C},
        $\x{354D},
        $\x{354E},
        $\x{354F},
        $\x{3550},
        $\x{3551},
        $\x{3552},
        $\x{3553},
        $\x{3554},
        $\x{3555},
        $\x{3556},
        $\x{3557},
        $\x{3558},
        $\x{3559},
        $\x{355A},
        $\x{3561},
        $\x{3562},
        $\x{3563},
        $\x{3564},
        $\x{3565},
        $\x{3566},
        $\x{3567},
        $\x{3568},
        $\x{3569},
        $\x{356A},
        $\x{356B},
        $\x{356C},
        $\x{356D},
        $\x{356E},
        $\x{356F},
        $\x{3570},
        $\x{3571},
        $\x{3572},
        $\x{3573},
        $\x{3574},
        $\x{3575},
        $\x{3576},
        $\x{3577},
        $\x{3578},
        $\x{3579},
        $\x{357A},
        $\x{3530},
        $\x{3531},
        $\x{3532},
        $\x{3533},
        $\x{3534},
        $\x{3535},
        $\x{3536},
        $\x{3537},
        $\x{3538},
        $\x{3539},
        $\x{352D},
        $\x{355F},
        $\x{3641},
        $\x{3642},
        $\x{3643},
        $\x{3644},
        $\x{3645},
        $\x{3646},
        $\x{3647},
        $\x{3648},
        $\x{3649},
        $\x{364A},
        $\x{364B},
        $\x{364C},
        $\x{364D},
        $\x{364E},
        $\x{364F},
        $\x{3650},
        $\x{3651},
        $\x{3652},
        $\x{3653},
        $\x{3654},
        $\x{3655},
        $\x{3656},
        $\x{3657},
        $\x{3658},
        $\x{3659},
        $\x{365A},
        $\x{3661},
        $\x{3662},
        $\x{3663},
        $\x{3664},
        $\x{3665},
        $\x{3666},
        $\x{3667},
        $\x{3668},
        $\x{3669},
        $\x{366A},
        $\x{366B},
        $\x{366C},
        $\x{366D},
        $\x{366E},
        $\x{366F},
        $\x{3670},
        $\x{3671},
        $\x{3672},
        $\x{3673},
        $\x{3674},
        $\x{3675},
        $\x{3676},
        $\x{3677},
        $\x{3678},
        $\x{3679},
        $\x{367A},
        $\x{3630},
        $\x{3631},
        $\x{3632},
        $\x{3633},
        $\x{3634},
        $\x{3635},
        $\x{3636},
        $\x{3637},
        $\x{3638},
        $\x{3639},
        $\x{362D},
        $\x{365F},
        $\x{3741},
        $\x{3742},
        $\x{3743},
        $\x{3744},
        $\x{3745},
        $\x{3746},
        $\x{3747},
        $\x{3748},
        $\x{3749},
        $\x{374A},
        $\x{374B},
        $\x{374C},
        $\x{374D},
        $\x{374E},
        $\x{374F},
        $\x{3750},
        $\x{3751},
        $\x{3752},
        $\x{3753},
        $\x{3754},
        $\x{3755},
        $\x{3756},
        $\x{3757},
        $\x{3758},
        $\x{3759},
        $\x{375A},
        $\x{3761},
        $\x{3762},
        $\x{3763},
        $\x{3764},
        $\x{3765},
        $\x{3766},
        $\x{3767},
        $\x{3768},
        $\x{3769},
        $\x{376A},
        $\x{376B},
        $\x{376C},
        $\x{376D},
        $\x{376E},
        $\x{376F},
        $\x{3770},
        $\x{3771},
        $\x{3772},
        $\x{3773},
        $\x{3774},
        $\x{3775},
        $\x{3776},
        $\x{3777},
        $\x{3778},
        $\x{3779},
        $\x{377A},
        $\x{3730},
        $\x{3731},
        $\x{3732},
        $\x{3733},
        $\x{3734},
        $\x{3735},
        $\x{3736},
        $\x{3737},
        $\x{3738},
        $\x{3739},
        $\x{372D},
        $\x{375F},
        $\x{3841},
        $\x{3842},
        $\x{3843},
        $\x{3844},
        $\x{3845},
        $\x{3846},
        $\x{3847},
        $\x{3848},
        $\x{3849},
        $\x{384A},
        $\x{384B},
        $\x{384C},
        $\x{384D},
        $\x{384E},
        $\x{384F},
        $\x{3850},
        $\x{3851},
        $\x{3852},
        $\x{3853},
        $\x{3854},
        $\x{3855},
        $\x{3856},
        $\x{3857},
        $\x{3858},
        $\x{3859},
        $\x{385A},
        $\x{3861},
        $\x{3862},
        $\x{3863},
        $\x{3864},
        $\x{3865},
        $\x{3866},
        $\x{3867},
        $\x{3868},
        $\x{3869},
        $\x{386A},
        $\x{386B},
        $\x{386C},
        $\x{386D},
        $\x{386E},
        $\x{386F},
        $\x{3870},
        $\x{3871},
        $\x{3872},
        $\x{3873},
        $\x{3874},
        $\x{3875},
        $\x{3876},
        $\x{3877},
        $\x{3878},
        $\x{3879},
        $\x{387A},
        $\x{3830},
        $\x{3831},
        $\x{3832},
        $\x{3833},
        $\x{3834},
        $\x{3835},
        $\x{3836},
        $\x{3837},
        $\x{3838},
        $\x{3839},
        $\x{382D},
        $\x{385F},
        $\x{3941},
        $\x{3942},
        $\x{3943},
        $\x{3944},
        $\x{3945},
        $\x{3946},
        $\x{3947},
        $\x{3948},
        $\x{3949},
        $\x{394A},
        $\x{394B},
        $\x{394C},
        $\x{394D},
        $\x{394E},
        $\x{394F},
        $\x{3950},
        $\x{3951},
        $\x{3952},
        $\x{3953},
        $\x{3954},
        $\x{3955},
        $\x{3956},
        $\x{3957},
        $\x{3958},
        $\x{3959},
        $\x{395A},
        $\x{3961},
        $\x{3962},
        $\x{3963},
        $\x{3964},
        $\x{3965},
        $\x{3966},
        $\x{3967},
        $\x{3968},
        $\x{3969},
        $\x{396A},
        $\x{396B},
        $\x{396C},
        $\x{396D},
        $\x{396E},
        $\x{396F},
        $\x{3970},
        $\x{3971},
        $\x{3972},
        $\x{3973},
        $\x{3974},
        $\x{3975},
        $\x{3976},
        $\x{3977},
        $\x{3978},
        $\x{3979},
        $\x{397A},
        $\x{3930},
        $\x{3931},
        $\x{3932},
        $\x{3933},
        $\x{3934},
        $\x{3935},
        $\x{3936},
        $\x{3937},
        $\x{3938},
        $\x{3939},
        $\x{392D},
        $\x{395F},
        $\x{2D41},
        $\x{2D42},
        $\x{2D43},
        $\x{2D44},
        $\x{2D45},
        $\x{2D46},
        $\x{2D47},
        $\x{2D48},
        $\x{2D49},
        $\x{2D4A},
        $\x{2D4B},
        $\x{2D4C},
        $\x{2D4D},
        $\x{2D4E},
        $\x{2D4F},
        $\x{2D50},
        $\x{2D51},
        $\x{2D52},
        $\x{2D53},
        $\x{2D54},
        $\x{2D55},
        $\x{2D56},
        $\x{2D57},
        $\x{2D58},
        $\x{2D59},
        $\x{2D5A},
        $\x{2D61},
        $\x{2D62},
        $\x{2D63},
        $\x{2D64},
        $\x{2D65},
        $\x{2D66},
        $\x{2D67},
        $\x{2D68},
        $\x{2D69},
        $\x{2D6A},
        $\x{2D6B},
        $\x{2D6C},
        $\x{2D6D},
        $\x{2D6E},
        $\x{2D6F},
        $\x{2D70},
        $\x{2D71},
        $\x{2D72},
        $\x{2D73},
        $\x{2D74},
        $\x{2D75},
        $\x{2D76},
        $\x{2D77},
        $\x{2D78},
        $\x{2D79},
        $\x{2D7A},
        $\x{2D30},
        $\x{2D31},
        $\x{2D32},
        $\x{2D33},
        $\x{2D34},
        $\x{2D35},
        $\x{2D36},
        $\x{2D37},
        $\x{2D38},
        $\x{2D39},
        $\x{2D2D},
        $\x{2D5F},
        $\x{5F41},
        $\x{5F42},
        $\x{5F43},
        $\x{5F44},
        $\x{5F45},
        $\x{5F46},
        $\x{5F47},
        $\x{5F48},
        $\x{5F49},
        $\x{5F4A},
        $\x{5F4B},
        $\x{5F4C},
        $\x{5F4D},
        $\x{5F4E},
        $\x{5F4F},
        $\x{5F50},
        $\x{5F51},
        $\x{5F52},
        $\x{5F53},
        $\x{5F54},
        $\x{5F55},
        $\x{5F56},
        $\x{5F57},
        $\x{5F58},
        $\x{5F59},
        $\x{5F5A},
        $\x{5F61},
        $\x{5F62},
        $\x{5F63},
        $\x{5F64},
        $\x{5F65},
        $\x{5F66},
        $\x{5F67},
        $\x{5F68},
        $\x{5F69},
        $\x{5F6A},
        $\x{5F6B},
        $\x{5F6C},
        $\x{5F6D},
        $\x{5F6E},
        $\x{5F6F},
        $\x{5F70},
        $\x{5F71},
        $\x{5F72},
        $\x{5F73},
        $\x{5F74},
        $\x{5F75},
        $\x{5F76},
        $\x{5F77},
        $\x{5F78},
        $\x{5F79},
        $\x{5F7A},
        $\x{5F30},
        $\x{5F31},
        $\x{5F32},
        $\x{5F33},
        $\x{5F34},
        $\x{5F35},
        $\x{5F36},
        $\x{5F37},
        $\x{5F38},
        $\x{5F39},
        $\x{5F2D},
        $\x{5F5F}
    }).
